/******************************************************************************
 * (C) Copyright 2000 by Agilent Technologies GmbH. All rights reserved.      *
 ******************************************************************************/


/* ---------------------------------------------------------------
* File: xsession.c
*       organize the connection and some more management functions
* -----------------------------------------------------------------*/

#ifdef BEST_DEBUG_XSESSION
/* Always use it with two bracket pairs, e.g.
   DBG_OUT(("A=%lu B=%lu C=%lu\n",a,b,c));
   This makes the preprocessor handle it as only one
   parameter (containing multiple parameters enclosed
   in brackets).
*/
#define DBG_OUT(a)          BESTX_PRINTF a
#else
#define DBG_OUT(a)          (void) (0)
#endif

#ifdef BEST_FIRMWARE

#include <xcapitype.h>
#include <xhal.h>

#else

#include <xtypedef.h>

#include <xregcons.h>
#include <xdynamic.h>
#include <xerrcapi.h>

#include <xio.h>
#include <xpci.h>
#include <xcmd.h>
#include <timeout.h>
#include <xidentos.h>

#if !defined(AIX) && !defined(_AIX)
#include <xepp.h>
#include <xserial.h>
#include <xhif.h>
#include <xusb.h>
#endif

#ifdef WIN32
#   include <xpci32rg.h>
#endif

#endif

#include <xregxdir.h>
#include <xregx10.h>
#include <xregx11.h>
#include <xaddrmap.h>
#include <xmephist.h> /* for Address of resourceLockRegister etc.*/

#include <xstrutil.h>

/* This is for production only */
bx_porttype EXPORT BestXGetportProduction(bx_handletype handle)
{
    return bx_handlearray[handle].port;
}

bx_portnumtype EXPORT BestXGetportnumProduction(bx_handletype handle)
{
    return bx_handlearray[handle].portnumber;
}

bx_int32 EXPORT BestXGetentered_portProduction(bx_handletype handle)
{
    return bx_handlearray[handle].entered_port;
}

/* Prototypes for resource locking functions */
static bx_errtype BestXResourceLockInit(bx_handletype handle);
static bx_errtype BestXResourceLockDeterminePort(bx_handletype handle,
                                                 bx_int32 *lock_port);
static bx_errtype BestXResourceLockIncreaseCounter(bx_handletype,
                                                   bx_resourcetype resource);
static bx_errtype BestXResourceLockDecreaseCounter(bx_handletype handle,
                                                   bx_resourcetype resource);
static bx_errtype BestXResourceLockFunction(bx_handletype handle,
                                           bx_int32 This_Port_Mask,
                                           bx_int32 Other_Port_Mask,
                                           bx_resourcetype resource);
static bx_errtype BestXResourceUnlockFunction(bx_handletype handle,
                                              bx_int32 Port_Mask,
                                              bx_resourcetype resource);

static bx_errtype BestXDirectRegPCIWriteEx(
  bx_handletype handle,
  bx_int32 dir_addr,
  bx_int32 regsize,    
  bx_int32 reg_value
);

/* prototypes for data buffer access */
static void DataBufferDelete(bx_handletype handle);
static void DataBufferResize(bx_handletype handle, bx_int32 size);

#ifndef BEST_FIRMWARE

bx_int32 GlobalBaseAddr;

/* -------------------------------------------------------------------------
* bx_handlearray is the global array that holds all information necessary to
* access hardware and to connect to all needed ports.
* The elements in the bx_handlestype struct are:
*
* port, portnumber, entered_port, is_open, is_connected, is_reserved
* hwinfo, param (RS232 => baudrate), regwidth, timeouts,
* 4 function ptrs, dynamic capability list ptr, perfboard_gapmode, lasterr
*
* -------------------------------------------------------------------------
*/

#define LAST_ERR_INIT   {{0}, {0}, BX_E_BAD_HANDLE, {0}}

/* initial handle value */
#if defined(AIX) || defined(_AIX)
#define HANDLE_INIT {                                                    \
  BX_PORT_RS232,                /* port */                               \
  INVALID_OS_HANDLE,            /* portnumber */                         \
  BX_PORT_COM1,                 /* entered_port */                       \
  0, 0, 0,                      /* is_open, is_connected, is_reserved */ \
  BX_HW_INFO_UNKNOWN,           /* hwinfo */                             \
  9600UL, 0, NO_TIMEOUTS,       /* param, regwidth, timeouts */          \
  NULL, NULL, NULL, NULL,       /* various callback fn pointers */       \
  NULL,                         /* capable */                            \
  0,                            /* gapmode */                            \
  LAST_ERR_INIT,                /* lasterr */                            \
  NULL,                         /* db */                                 \
  0xffffffff,                   /* CardIsInProgrammingMode */            \
  0UL,                          /* m_bufferSize */                       \
  NULL,                         /* m_pBuffer */                          \
  NULL                          /* This is where we store the AIX handle */
}
#else
#define HANDLE_INIT {                                                    \
  BX_PORT_RS232,                /* port */                               \
  INVALID_OS_HANDLE,            /* portnumber */                         \
  BX_PORT_COM1,                 /* entered_port */                       \
  0, 0, 0,                      /* is_open, is_connected, is_reserved */ \
  BX_HW_INFO_UNKNOWN,           /* hwinfo */                             \
  9600UL, 0, NO_TIMEOUTS,       /* param, regwidth, timeouts */          \
  NULL, NULL, NULL, NULL,       /* various callback fn pointers */       \
  NULL,                         /* capable */                            \
  0,                            /* gapmode */                            \
  LAST_ERR_INIT,                /* lasterr */                            \
  NULL,                         /* db */                                 \
  0xffffffff,                   /* CardIsInProgrammingMode */            \
  0UL,                          /* m_bufferSize */                       \
  NULL,                         /* m_pBuffer */                          \
}
#endif

#define HANDLE_INIT_ERR \
{BX_PORT_OFFLINE, INVALID_OS_HANDLE, 0, 0, 0, 0, \
  BX_HW_INFO_UNKNOWN, 0, 0, NO_TIMEOUTS, \
NULL, NULL, NULL, NULL, NULL, 0, LAST_ERR_INIT,NULL,0xffffffff}

#define FOUR_H_INITS HANDLE_INIT, HANDLE_INIT, HANDLE_INIT, HANDLE_INIT

bx_handlestype bx_handlearray[BX_MAXHANDLES + 1] = {
  FOUR_H_INITS, FOUR_H_INITS, FOUR_H_INITS, FOUR_H_INITS,
    FOUR_H_INITS, FOUR_H_INITS, FOUR_H_INITS, FOUR_H_INITS,
  FOUR_H_INITS, FOUR_H_INITS, FOUR_H_INITS, FOUR_H_INITS,
    FOUR_H_INITS, FOUR_H_INITS, FOUR_H_INITS, FOUR_H_INITS,
  FOUR_H_INITS, FOUR_H_INITS, FOUR_H_INITS, FOUR_H_INITS,
    FOUR_H_INITS, FOUR_H_INITS, FOUR_H_INITS, FOUR_H_INITS,
  FOUR_H_INITS, FOUR_H_INITS, FOUR_H_INITS, FOUR_H_INITS,
    FOUR_H_INITS, FOUR_H_INITS, FOUR_H_INITS, FOUR_H_INITS,
  FOUR_H_INITS, FOUR_H_INITS, FOUR_H_INITS, FOUR_H_INITS,
    FOUR_H_INITS, FOUR_H_INITS, FOUR_H_INITS, FOUR_H_INITS,
  FOUR_H_INITS, FOUR_H_INITS, FOUR_H_INITS, FOUR_H_INITS,
    FOUR_H_INITS, FOUR_H_INITS, FOUR_H_INITS, FOUR_H_INITS,
  FOUR_H_INITS, FOUR_H_INITS, FOUR_H_INITS, FOUR_H_INITS,
    FOUR_H_INITS, FOUR_H_INITS, FOUR_H_INITS, FOUR_H_INITS,
  FOUR_H_INITS, FOUR_H_INITS, FOUR_H_INITS, FOUR_H_INITS,
    FOUR_H_INITS, FOUR_H_INITS, FOUR_H_INITS, FOUR_H_INITS,

    /* this next entry (at index == BX_MAXHANDLES) is used to init a user's handle
    * to BX_MAXHANDLES so that we can detect an error if the open fails but they
    * don't check the return value. */
    HANDLE_INIT_ERR
};

#if !defined(AIX) && !defined(_AIX)
/* to reset any element of the handle array to initial values... */
static const bx_handlestype xhandle_null = HANDLE_INIT;
#else
/* We need to modify this in tu_open in bx_dd_interface so it can't be static const */
bx_handlestype xhandle_null = HANDLE_INIT; /* extern in xtypedef.h */
#endif

#define BX_DEF_CONNECT_TIMEOUT    1000

#define FOUR_TO_INITS   BX_DEF_CONNECT_TIMEOUT, BX_DEF_CONNECT_TIMEOUT, \
BX_DEF_CONNECT_TIMEOUT, BX_DEF_CONNECT_TIMEOUT

bx_int32 xconnect_timeout_array[BX_MAXHANDLES + 1] = {
  FOUR_TO_INITS, FOUR_TO_INITS, FOUR_TO_INITS, FOUR_TO_INITS,
    FOUR_TO_INITS, FOUR_TO_INITS, FOUR_TO_INITS, FOUR_TO_INITS,
  FOUR_TO_INITS, FOUR_TO_INITS, FOUR_TO_INITS, FOUR_TO_INITS,
    FOUR_TO_INITS, FOUR_TO_INITS, FOUR_TO_INITS, FOUR_TO_INITS,
  FOUR_TO_INITS, FOUR_TO_INITS, FOUR_TO_INITS, FOUR_TO_INITS,
    FOUR_TO_INITS, FOUR_TO_INITS, FOUR_TO_INITS, FOUR_TO_INITS,
  FOUR_TO_INITS, FOUR_TO_INITS, FOUR_TO_INITS, FOUR_TO_INITS,
    FOUR_TO_INITS, FOUR_TO_INITS, FOUR_TO_INITS, FOUR_TO_INITS,
  FOUR_TO_INITS, FOUR_TO_INITS, FOUR_TO_INITS, FOUR_TO_INITS,
    FOUR_TO_INITS, FOUR_TO_INITS, FOUR_TO_INITS, FOUR_TO_INITS,
  FOUR_TO_INITS, FOUR_TO_INITS, FOUR_TO_INITS, FOUR_TO_INITS,
    FOUR_TO_INITS, FOUR_TO_INITS, FOUR_TO_INITS, FOUR_TO_INITS,
  FOUR_TO_INITS, FOUR_TO_INITS, FOUR_TO_INITS, FOUR_TO_INITS,
    FOUR_TO_INITS, FOUR_TO_INITS, FOUR_TO_INITS, FOUR_TO_INITS,
  FOUR_TO_INITS, FOUR_TO_INITS, FOUR_TO_INITS, FOUR_TO_INITS,
    FOUR_TO_INITS, FOUR_TO_INITS, FOUR_TO_INITS, FOUR_TO_INITS,
    BX_DEF_CONNECT_TIMEOUT
};


#if BX_MAXHANDLES > 256
#error Initialization of bx_handlearray,xconnect_timeout_array and DirectFP invalidated by increase in BX_MAXHANDLES
#endif

/* statics */
static bx_errtype BestXIdHardware(bx_handletype handle, bx_OpenType task);

/*
   This is for debug output of
   BestXDirectRegWrite/Read() functions
*/
static FILE * DirectFP[BX_MAXHANDLES]=
{
  NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,
  NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,
  NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,
  NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,
  NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,
  NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,
  NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,
  NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,
  NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,
  NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,
  NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,
  NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,
  NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,
  NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,
  NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,
  NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL
};

/* This is the report file, where each call to
   BestXDirectRegRead/Write() reports its HW-accesses
 */
#ifdef BEST_DEBUG

#define DIRECT_REPORT_FILE "c:\\bx_direct"

#endif /* BEST_DEBUG */

void BestXConnectTimeoutSet(bx_handletype handle, bx_int32 ms)
{
  assert(handle < BX_MAXHANDLES);
  xconnect_timeout_array[handle] = ms;
}


void BestXConnectTimeoutDefSet(bx_handletype handle)
{
  assert(handle < BX_MAXHANDLES);
  xconnect_timeout_array[handle] = BX_DEF_CONNECT_TIMEOUT;
}


/*---------------------------------------------------------
* oscheck checks the different operating systems and ensures that
* you do not run best under the wrong os
*---------------------------------------------------------*/
bx_ccharptrtype EXPORT BestXOSCheck()
{
  BX_OS_VERSION actualOS, targetOS;
  actualOS = GetOsVersion(NULL, &targetOS);

  if (actualOS != targetOS)
  {
    if (targetOS == OS_WIN2K && actualOS == OS_WIN95)
    {
      return NULL;
    }
    if (targetOS == OS_WIN95 && actualOS == OS_WIN2K)
    {
      return (char *) "This version of best.exe was compiled for WIN95. It will not run under WIN2K. Please install the WIN2K version of best.exe.";
    }
    if (targetOS == OS_WINNT && actualOS == OS_WIN95)
    {
      return (char *) "This version of best.exe was compiled for WINNT. It will not run under WIN95. Please install the WIN95 version of best.exe.";
    }
    if (targetOS == OS_WIN95 && actualOS == OS_WINNT)
    {
      return (char *) "This version of best.exe was compiled for WIN95. It will not run under WINNT. Please install the WINNT version of best.exe.";
    }
    if (targetOS == OS_WINNT && actualOS == OS_WIN2K)
    {
      return (char *) "This version of best.exe was compiled for WINNT. It will not run under WIN2K. Please install the WIN2K version of best.exe.";
    }
    if (targetOS == OS_WIN2K && actualOS == OS_WINNT)
    {
      return (char *) "This version of best.exe was compiled for WIN2K. It will not run under WINNT. Please install the WINNT version of best.exe.";
    }
    else
    {
      return (char *) "This version of best.exe was compiled for another operating system than it is running under. Please use the correct software version.";
    }
  }

  return NULL;
}


/*
////////////////////////////////////////////////////////////////////////////////////
// SCR; 5/22/97; Pretty radical changes when using the Win32 registry (multiple devices)
*/

/* BestXDevIdentifierGet moved to xmailbox.c */


/* --------------------------------------------------------------------------
* This helper function is separated because it's used by BestXOpenAdv()
* and one other function.
* -------------------------------------------------------------------------- */

bx_errtype BestXFirstFreeHandleGet(bx_handletype * pHandle)
{
  bx_handletype handle;

  for (handle = 0; handle < (bx_handletype) BX_MAXHANDLES; handle++)
  {
    if ((bx_handlearray[handle].is_open == 0) &&
      (bx_handlearray[handle].is_reserved == 0))
    {
      break;
    }
  }

  *pHandle = handle;

  /* no more free handles left ? */
  return(handle == BX_MAXHANDLES ? BX_E_NO_HANDLE_LEFT : BX_E_OK);
}


/******************************************************************************/
/* --------------------------------------------------------------------------
* Returns BX_E_OK and fBoardWasReset = 1 if the board was reset since Opening
* OR since the last call to BestXBasicIsBoardReset().
* fBoardWasReset is undefined if the return is NOT BX_E_OK.
* -------------------------------------------------------------------------- */

bx_errtype EXPORT BestXBasicIsBoardReset(
  bx_handletype handle,
  bx_int32 *fBoardWasReset)
{
  BX_DECLARE_FUNCNAME("BestXBasicIsBoardReset");

  bx_errtype err;
  BX_FCT_PARAM_NULL_POINTER_CHK(fBoardWasReset);

  {
    bx_int32 isReset;
    bx_int8 out_zw[OUT_RESET_CHECK];
    bx_int16 outsize = OUT_RESET_CHECK;

    if(BestXHasFirmware(handle))
    {
      err = BestXBasicCommand(handle, CMD_RESET_CHECK,
            NULL, IN_RESET_CHECK,out_zw, &outsize);
      (void) BestXStream2Long(&isReset, out_zw, 1);
      *fBoardWasReset = isReset != 0 ? 1 : 0;
    }
    else
    {
      err = BX_E_OK;
      *fBoardWasReset=0;
    }
  }

  BX_ERRETURN(err);
}


/* -------------------------------------------------------------------------*/
/* This is the classic open call, called from a user program or from the
* GUI. It is mapped onto BestXOpenAdv to have only one function that handles
* all open accesses. */
/* -------------------------------------------------------------------------*/

bx_errtype EXPORT BestXOpen(
  bx_handletype * pHandle,
  bx_porttype port,
  bx_int32 portnum)
{
  /* Do not use BX_ERRETURN here !! CZ */
  return BestXOpenAdv(pHandle, port, portnum, NULL, BX_OPEN_STANDARD,1 /* do soft force */,NULL);
}

/* -------------------------------------------------------------------------
   Tries to open card as fast as possible:
   No version check.
   No license checking (must have all licenses).
   Assumes card is in BIOS mode.
   This function is mainly used to yield a faster BestXOpen() for the PCI port
   when calling BestXOpen() on a card with firmware many times.
   Recommendation: 
     You should call the standard BestXOpen() when opening the
     card for the first time and only use this function for successive calls
     to the same card.
   Not officially supported.
   -------------------------------------------------------------------------*/

bx_errtype EXPORT BestXFastOpen(
  bx_handletype * pHandle,
  bx_porttype port,
  bx_int32 portnum)
{
  /* Do not use BX_ERRETURN here !! CZ */
  return BestXOpenAdv(pHandle, port, portnum, NULL, BX_OPEN_FAST,1 /* do soft force */,NULL);
}

/* --------------------------------------------------------------------------
* Use this open call.  It is more advanced and can handle several different
* types of open (with or without license, online or offline) and can even
* take a license number to enable licenses within the firmware.
* -------------------------------------------------------------------------- */

bx_errtype EXPORT BestXOpenAdv(
  bx_handletype * pUsrHandle,
  bx_porttype port,
  bx_int32 portnum,
  bx_int32 * license_code,
  bx_OpenType task,
  bx_int32 SoftForce, /* 1=force (after trying to sync),
                        0=no forcing,i.e. function may return an error,
                        if PCI clock is slow or missing */

  bx_charptrtype filename  /* FX2 firmware to download */                      
)
{
  BX_DECLARE_FUNCNAME("BestXOpenAdv");

  BX_TRY_VARS;
  bx_portnumtype OsHandle = INVALID_OS_HANDLE;
  bx_handletype handle = BX_INVALID_HANDLE;
  bx_int32 HPSlotId = 0;         /* used for PCI and FastHIF */
  int j;

  BX_TRY_BEGIN
  {
    BX_TRY_FCT_PARAM(1, port>BX_PORT_PRODUCTION);

    /* Bios switch task looks whether *pUsrHandle is a reserved handle. If
     * this is the case it keeps the same handle, otherwise it follows the
     * standard procedure */
    if ((task == BX_OPEN_BIOS_SWITCH) || (task == BX_OPEN_STANDARD_RESERVED))
    {
      handle = *pUsrHandle;

      if (handle < BX_MAXHANDLES)
      {
        if ((!bx_handlearray[handle].is_open) &&
            bx_handlearray[handle].is_reserved)
        {
          goto handle_label;
        }
      }
    }

    /* BX_MAXHANDLES is a valid index in our handle array but will produce an
     * error in almost any program....thus we're ok if the open fails but the
     * caller ignores the returned error code */
    *pUsrHandle = BX_MAXHANDLES;

    /* run through the handle array and look for a free handle */
    BX_TRY(BestXFirstFreeHandleGet(&handle));

    handle_label:

    bx_handlearray[handle].is_reserved = 0;
    
    if (task == BX_OPEN_STANDARD_RESERVED)
    {
      task = BX_OPEN_LICENSE; /* HL, 06-Juli-2000 changed this for SVP use */
    }

    /* Note; we MUST re-initialize prior to continuing so that a previously
     * interrupted open cannot leave us in an unknown state. */
    /* Note2: if in AIX, xhandle_null contains the AIX_handle ptr and is therefore
     * only somewhat null */
    bx_handlearray[handle] = xhandle_null;
    bx_handlearray[handle].OpenType = task; /* needed for BX_OPEN_FAST */

    if (task == BX_OPEN_RESERVE)
    {
      bx_handlearray[handle].is_reserved = 1;
      *pUsrHandle = handle;
      BX_ERRETURN(BX_E_OK);
    }

    /* these may change depending on the port */
    bx_handlearray[handle].port = port;
    bx_handlearray[handle].entered_port = portnum;
    /* We assign pUsrHandle below since the card will soon be opened                 *
     * If the program receives a SIGQUIT or similar signal, this will cause          *
     * the card to close.  If this is left below, there is a lot of other operations *
     * that will happen before pUsrHandle is assigned which could cause the card not *
     * to close properly                   *
     * If for some reason the card does not open, this will not matter as there are  *
     * other checks for the AIXhandle and mdd_fd before attempting to close the card */ 
#if _AIX
    *pUsrHandle = handle;
#endif

    switch (port)
    {
      case BX_PORT_OFFLINE:
        /* portnum codes hardware and license in case of OFFLINE port */
        OsHandle = (bx_portnumtype) portnum;
        break;

#if !defined(AIX) && !defined(_AIX)  
      case BX_PORT_RS232:
        OsHandle = BestXOpenCOMInitial((int) portnum, 9600UL);
        BX_TRY_FAIL(INVALID_OS_HANDLE == OsHandle ? BX_E_RS232_OPEN : BX_E_OK);
        break;

      case BX_PORT_PARALLEL:
        OsHandle = BestXOpenParallel((int) portnum);
        BX_TRY_FAIL(INVALID_OS_HANDLE == OsHandle ? BX_E_USB_OPEN : BX_E_OK);
        /* TODO, if needed: add errormessage BX_E_PARALLEL_OPEN and use here instead */
        break;

      case BX_PORT_FASTHIF:
        BX_TRY(BestXOpenHIF((bx_int32) portnum, &HPSlotId, &OsHandle));
        BX_TRY_FAIL(INVALID_OS_HANDLE == OsHandle ? BX_E_HIF_OPEN : BX_E_OK);
/*         bx_handlearray[handle].entered_port = HPSlotId; */
        break;

      case BX_PORT_USB:
        BX_TRY(BestXOpenUSB((bx_int32) portnum, &OsHandle,filename));
        BX_TRY_FAIL(INVALID_OS_HANDLE == OsHandle ? BX_E_USB_OPEN : BX_E_OK);
        break;

#endif
#ifdef CUSTOM_OEM1
      case BX_PORT_OEM:
        BX_TRY(BestXOpenOEM((int) portnum, &(bx_handlearray[handle].OEMData)));
        BX_TRY_FAIL(INVALID_OS_HANDLE == bx_handlearray[handle].OEMData.SVFSFD ? BX_E_OEM_OPEN : BX_E_OK);

        BX_TRY(BestXOpenPCI((int) portnum, &OsHandle));
        if( OsHandle == INVALID_OS_HANDLE )
        {
            BX_TRY(BestXCloseOEM( bx_handlearray[handle].OEMData ) ) ;
        }
        BX_TRY_FAIL(INVALID_OS_HANDLE == OsHandle ? BX_E_OEM_OPEN : BX_E_OK);
        break;
#endif


      case BX_PORT_PCI_CONF:
#if defined(AIX) || defined(_AIX) /* We want handlearray index */
        portnum = (bx_handletype) handle;
#endif
        BX_TRY(BestXOpenPCI((int) portnum, &OsHandle));
        BX_TRY_FAIL(INVALID_OS_HANDLE == OsHandle ? BX_E_PCI_OPEN : BX_E_OK);
        break;

      case BX_PORT_PCI_IO:
        BX_TRY(BestXOpenIO((int) portnum, &OsHandle));
        BX_TRY_FAIL(INVALID_OS_HANDLE == OsHandle ? BX_E_IO_OPEN : BX_E_OK);
        bx_handlearray[handle].param = GlobalBaseAddr;
        break;

      case BX_PORT_PRODUCTION:
        if (task==BX_OPEN_USB23)
        {
          /* E2923 production via USB */
          BX_TRY(BestXOpenUSB((bx_int32) portnum, &OsHandle,filename));
          BX_TRY_FAIL(INVALID_OS_HANDLE == OsHandle ? BX_E_USB_OPEN : BX_E_OK);
        }
        else
        {
          /* E2922 production */
          OsHandle=(bx_portnumtype) *license_code;
          BX_TRY(BestXOpenProduction((bx_handletype)OsHandle));
        }
        break;

      default:
        return BX_E_INVALID_CASE;
    }                           /*lint !e788 not all enums used */

    /* this is the ONLY place in the entire API where we "open" */
    bx_handlearray[handle].portnumber = OsHandle;
    bx_handlearray[handle].is_open = 1;

    /* make sure we get into the catch... block later to close the port; in */
    /* case something fails later! */
    BX_TRY_PROGRESS (BX_E_OK);

    /* set timeouts...must be done before any reads/writes
     * from this point on a BestXClose() is required to release the handle
     */

    if(BX_OPEN_QUICKCHECK == task)
    {
      /* quickcheck open uses minimum timeouts.
       * we're pretty close to the limit...reduce at your own risk !
       */
      BESTTIMEOUTS tmrStruct;
      /* 0, 0, 100, 0, 50 works for the 2926 but NOT the 2925.
       * The 2925 needs between 500 and 600 mS to respond!
       */
      BestSetCommToStruct(&tmrStruct, 0, 0, 700, 0, 50);
      BX_TRY_PROGRESS(BestXPortTimeoutSet(handle, &tmrStruct, TIMEOUTS_SET));
      BestXConnectTimeoutSet(handle, 100);
    }
    else
    {
      BX_TRY_PROGRESS(BestXPortTimeoutSet(handle, NULL, TIMEOUTS_SET_DEF));
      BestXConnectTimeoutDefSet(handle);
    }

      /* identify hardware...this includes determination of the register set,
       * model, protocol, version etc.
       */
    BX_TRY(BestXIdHardware(handle, task));

    /* don't bother with dynamic cap's if we're just checking.  this
       * doesn't actually save a lot of time (the I/O is absolutely
       * the governing factor) but why do all those mallocs if we
       * don't need to?  */
    if(BX_OPEN_QUICKCHECK != task)
    {
      /* create the dynamic capabilities */
      if (task != BX_OPEN_BIOS_SWITCH)
      {
        /* FaustLoadError set to 1, if:
          - Not OFFLINE AND
          - FPGA hardware AND
          - Faust FPGA image could not be loaded
         */
        if (BestXHasFaustFPGA(handle))
        {
          /* Faust FPGA based HW (30A, 23F) => COCO available */
          /* Check, wether Faust has failed to load successfully */
          bx_int32 value=0;
          DBG_OUT (("Checking COCO status..."));
          BX_TRY(BestXDirectRegRead (handle, BX_REG_COCO_STATUS,2,&value));
          if (port==BX_PORT_OFFLINE || value&0x0800)
          {
            /* Faust loaded successfully */
            bx_handlearray[handle].FaustLoadError=0;
            DBG_OUT (("OK\n"));
          }
          else
          {
            /* Faust not loaded successfully (hw-update or production) */
            bx_handlearray[handle].FaustLoadError=1;
            DBG_OUT (("ERROR (Faust not loaded)\n"));
          }
        }
        else
        {
          /* ASIC based HW */
          bx_handlearray[handle].FaustLoadError=0;
        }

        /* Calls BestXIsMode2(), which accesses Faust (-> FaustLoadError needed here !) */
        /* This also reads the capabilities from the card and inits
           handle->capable
         */
        DBG_OUT (("BestXCapaInit..."));
        BX_TRY(BestXCapaInit(handle));
        DBG_OUT (("OK\n"));

        if ( (bx_handlearray[handle].capable->capa_code[0] & BX_CAPABILITY_CAPI) != 0 )
        {
          /* Only initialize if we have CAPI license! */

          if (BestXFaustLoadError(handle))
          {
            /* Faust not loaded for some reason, we allow BestXOpen to work,
              in oder todo e.g. a HW update and for production purposes.
              Next access to a Faust register will return an error */
          }
          else
          {
            /* Faust successfully loaded; we usually go here.
               These functions contain Faust accesses !!           
               BestXBoardInit():
               Switch card to runmode, init handle->CardIsInProgrammingMode,
               clear retry bit and read props and memories from card.
               This needs to be called after CapaInit() and DBInit() !!
            */
            DBG_OUT (("BestXDBInit..."));
            BX_TRY(BestXDBInit(handle));
            DBG_OUT (("OK\n"));
            DBG_OUT (("BestXBoardInit..."));
            BX_TRY(BestXBoardInit(handle,SoftForce));
            DBG_OUT (("OK\n"));
            BX_TRY(BestXResourceLockInit(handle));
          }
        }
      } /* task!=BX_OPEN_BIOS_SWITCH */

      /* copy the license code, so we know what license we have and what not */
      if ((task == BX_OPEN_LICENSE) && (license_code))
      {
        for (j = 0; j < 3; ++j)
        {
          license_code[j] = bx_handlearray[handle].capable->capa_code[j];
        }
      }
    }

#ifndef _AIX
    /* the ONLY successful open */
    *pUsrHandle = handle;
#endif
  }

  BX_TRY_CATCH
  {
    BX_TRY_PASSED
      {
        BestXLastErrorStore(handle, BX_TRY_RET);
        (void) BestXClose(handle);
        *pUsrHandle = BX_INVALID_HANDLE;

        /* Do not use BX_ERRETURN here!! Handleless error handling will get
         * confused. CZ */
        return (BX_TRY_RET);
      }
  }

  /* just in case we changed this */
  if(BX_OPEN_QUICKCHECK == task)
  {
    BestXConnectTimeoutDefSet(handle);
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*----------------------------------------------------------------------
* BestXReopen does a close/open on a stalled card
*----------------------------------------------------------------------*/
bx_errtype EXPORT
BestXReopen(bx_handletype handle, bx_OpenType task)
{
#if !defined(AIX) && !defined(_AIX)
  BX_TRY_VARS_NO_PROG;
  BX_TRY_BEGIN
  {
    bx_porttype port;
    bx_int32 portnum;
    bx_int32 baudrate;

    BX_TRY_HANDLE;

    port = bx_handlearray[handle].port;
    portnum = bx_handlearray[handle].entered_port;

    /* release connection, just to make sure */
    BestXReleaseConnection (handle);

    if (port == BX_PORT_RS232)
    {
      /* save baudrate for later */
      baudrate = bx_handlearray[handle].param;

      /* change COM port settings to default baudrate */
      BX_TRY(BestXSerInit(bx_handlearray[handle].portnumber, 9600UL));
      bx_handlearray[handle].param = 9600UL; /* current baudrate at serial */
    }

    /* establish new connection */
    BX_TRY(BestXOpenConnection(handle, xconnect_timeout_array[handle]));

    if (port == BX_PORT_RS232)
    {
      /* restore old baudrate */
      BX_TRY (BestXRS232BaudRateSet (handle, baudrate));
    }
  }

  BX_ERRETURN (BX_TRY_RET);
#else
  return (BX_E_INVALID_CASE);
#endif
 
}

/* --------------------------------------------------------------------------
* The default values for hardware identification.
* Note that the handle[].hwinfo is reset in BestXOpenAdv() and BestXCloseAdv()
* to BX_HW_INFO_UNKNOWN
* -------------------------------------------------------------------------- */

/*
the bx_hwinfotype structure;

  bx_int8ptr     product_string;
  bx_hwtype      hw;             compatibility...old bx_handlearray[].hw
  bx_hwseriestype hwseries;      hardware series (no deep id)
  bx_regfiletype regfile;        compatibility...old bx_handlearray[].regfile
  bx_int32 hwbitmask;
*/

/****IMPORTANT******
* Maintain these defines if you change product strings or add a hardware
* definition bit
*******************/

/* NOTE; These are deliberately defined here so that they cannot
* be directly accessed outside of this file.
* Use the set of functions defined below (search HARDWARE CHARACTERISTICS)
* to set/check these hardware characteristics.
*/

#define BX_HWDEF_HAS_FHIF      (0x01UL << 0)
#define BX_HWDEF_HAS_ICHI      (0x01UL << 1)
#define BX_HWDEF_IS_DEEP       (0x01UL << 2)
#define BX_HWDEF_IS_CORE       (0x01UL << 3)
#define BX_HWDEF_IS_BLKMODE    (0x01UL << 4)
#define BX_HWDEF_HAS_PERF      (0x01UL << 5)
#define BX_HWDEF_IS_COMPACT    (0x01UL << 6)
#define BX_HWDEF_HAS_ICHI20    (0x01UL << 7)
#define BX_HWDEF_IS_DIRECTMODE (0x01UL << 8)
#define BX_HWDEF_HAS_FIRMWARE  (0x01UL << 9)
#define BX_HWDEF_HAS_FAUST     (0x01UL << 10)
#define BX_HWDEF_HAS_MEPHISTO  (0x01UL << 11)
#define BX_HWDEF_HAS_FAUSTFPGA (0x01UL << 12)

/* just to save a little space */
#define BX_HWDEF_HAS_FHIF_ICHI_PERF  (BX_HWDEF_HAS_FHIF | BX_HWDEF_HAS_ICHI | BX_HWDEF_HAS_PERF)

static const bx_hwinfotype hw_info_defs[] =
{
  {
    "E2929A",BX_HW_E2929A,BX_SERIES_E2929A,
      0x2929, BX_HWDEF_HAS_FHIF_ICHI_PERF |BX_HWDEF_HAS_FIRMWARE | BX_HWDEF_HAS_MEPHISTO
  },
  {
    "E2929A_DEEP",BX_HW_E2929A_DEEP,BX_SERIES_E2929A,
      0x2929, BX_HWDEF_HAS_FHIF_ICHI_PERF | BX_HWDEF_IS_DEEP | BX_HWDEF_HAS_FIRMWARE  | BX_HWDEF_HAS_MEPHISTO
  },
  {
    "E2922A",BX_HW_E2922A,BX_SERIES_E2922A,
      0x2922, BX_HWDEF_HAS_ICHI  | BX_HWDEF_HAS_MEPHISTO
  },
  {
    "E2929B",BX_HW_E2929B,BX_SERIES_E2929B,
      0x2929, BX_HWDEF_HAS_FHIF_ICHI_PERF |BX_HWDEF_HAS_FIRMWARE | BX_HWDEF_HAS_MEPHISTO
  },
  {
    "E2929B_DEEP",BX_HW_E2929B_DEEP,BX_SERIES_E2929B,
      0x2929, BX_HWDEF_HAS_FHIF_ICHI_PERF | BX_HWDEF_IS_DEEP | BX_HWDEF_HAS_FIRMWARE | BX_HWDEF_HAS_MEPHISTO
  },
  {
    "E2922B",BX_HW_E2922B,BX_SERIES_E2922B,
      0x2922, BX_HWDEF_HAS_ICHI | BX_HWDEF_HAS_MEPHISTO
  },
  {
    "E2930A",BX_HW_E2930A,BX_SERIES_E2930A,
      0x2930, BX_HWDEF_HAS_FHIF_ICHI_PERF | BX_HWDEF_IS_DEEP | BX_HWDEF_HAS_FIRMWARE | BX_HWDEF_HAS_FAUST | BX_HWDEF_HAS_FAUSTFPGA
  },
  {
    "E2923A",BX_HW_E2923A,BX_SERIES_E2923A,
      0x2923, BX_HWDEF_HAS_ICHI  | BX_HWDEF_HAS_FAUST
  },
  {
    "E2930B",BX_HW_E2930B,BX_SERIES_E2930B,
      0x2930, BX_HWDEF_HAS_FHIF_ICHI_PERF | BX_HWDEF_IS_DEEP | BX_HWDEF_HAS_FIRMWARE  | BX_HWDEF_HAS_FAUST
  },
  { 
    "E2923B",BX_HW_E2923B,BX_SERIES_E2923B,
      0x2923, BX_HWDEF_HAS_ICHI  | BX_HWDEF_HAS_FAUST
  },
  { 
    "E2923F",BX_HW_E2923F,BX_SERIES_E2923F,
      0x2923, BX_HWDEF_HAS_ICHI  | BX_HWDEF_HAS_FAUST | BX_HWDEF_HAS_FAUSTFPGA
  }
};

/*****************************************************************************
 * Mainly for SVP...string list of available cards
 *****************************************************************************/

#define NUM_HW_DEFAULTS (int)(sizeof(hw_info_defs) / sizeof(bx_hwinfotype))

/*---------------------------------------------------------------------------*
 * bx_errtype EXPORT BestXGetCardTypeCount(size_t * pCount)
 *
 * Purpose      :
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXGetCardTypeCount(size_t * pCount)
{
  *pCount = NUM_HW_DEFAULTS;
  return BX_E_OK;
}

/*---------------------------------------------------------------------------*
 * bx_errtype EXPORT BestXGetCardTypeAtIndex(
 *
 * Purpose      :
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXGetCardTypeAtIndex(
  bx_param_stringlisttype * pCardList,
  size_t idx
  )
{
  if ( idx  < NUM_HW_DEFAULTS )
  {
    pCardList->name = hw_info_defs[idx].product_string;
    pCardList->value = hw_info_defs[idx].hw;
  }
  else
  {
    assert(("Bad Index", 0));
    return BX_E_PARAM_OUT_OF_RANGE;
  }

  return BX_E_OK;
}


/* --------------------------------------------------------------------------
* Some functions need to know if a PCI device id is actually a BestX
* -------------------------------------------------------------------------- */

bx_bool EXPORT BestXIsPciDeviceIdBestX(bx_int32 dev_id)
{
  int i;

  for(i = 0; i < NUM_HW_DEFAULTS; i++)
  {
    if(dev_id == hw_info_defs[i].dev_id)
    {
      return 1;
    }
  }
  return 0;
}


/* --------------------------------------------------------------------------
* All hardware identification done here...bx_hwinfotype declared in typedefs.h
* -------------------------------------------------------------------------- */

/* In debug mode we still allow BX_OPEN_FAILSAFE to bypass the version checks.
* We also have a special task BX_OPEN_BIOS_SWITCH which prepares for a
* core-switch (and thus must skip the version check).
*/

#ifdef BEST_DEBUG
# define DO_VERSION_CHECK  \
  ((task != BX_OPEN_FAILSAFE) && \
  (task != BX_OPEN_BIOS_SWITCH) && \
  (task != BX_OPEN_QUICKCHECK) && \
  (BESTX_FOPEN("BestXFILE_NoVersionCheck","r")==NULL) /* allow skipping version check */ \
)
#else           /* BEST_DEBUG */
# define DO_VERSION_CHECK  \
  (task != BX_OPEN_BIOS_SWITCH) && \
  (task != BX_OPEN_FAST) && \
(task != BX_OPEN_QUICKCHECK)
#endif          /* BEST_DEBUG */


/*---------------------------------------------------------------------------*
 * static bx_errtype BestXIdHardware(bx_handletype handle, bx_OpenType task)
 *
 * Purpose  : identify hardware and check some version info
 *---------------------------------------------------------------------------*/
static bx_errtype BestXIdHardware(bx_handletype handle, bx_OpenType task)
{
  BX_TRY_VARS_NO_PROG;
  bx_handlestype *pHandle = &(bx_handlearray[handle]);
  int i;
  int fIsOffline = (BX_PORT_OFFLINE == pHandle->port);
  int fDoVerCheck = DO_VERSION_CHECK;
  char buf[30] = {0};
  char *info=NULL;
  bx_hwtype hw;
  bx_int32 value;

  bx_bool hasFirmware;

  BX_TRY_BEGIN
  {
    /* classify hardware...currently BX_HW_INFO_UNKNOWN */

    if (fIsOffline)
    {
      /* offline is special; the user combines the hardware and license into
       * a mask ... put into bx_handlearray[].portnumber in BestXOpenAdv().
       * We're only interested in hardware here. */

      hw = (bx_hwtype) (BX_CAPABILITY_HARDWARE_MASK &
      (bx_int32) pHandle->portnumber);

      for (i = 0; i < NUM_HW_DEFAULTS; i++)
      {
        if (hw == hw_info_defs[i].hw)
        {
          /* good one !! */
          pHandle->hwinfo = hw_info_defs[i];
          break;
        }
      }
    }
    else /* we're on-line */
    {
      /* get HW type from board ... note that RPRODUCT_STRING (0x02 in 2926
       * mode !!) is byte-size across all boards;
       * 2926-bios/core -> a real product string.
       * we can thus determine which card (and mode) we're talking to.
       */

      /* e.g. "E2929A" (s_product_string from root.c) */

      /* retrieval of hardware type is port-dependent: */
      switch (pHandle->port)
      {
        case BX_PORT_PRODUCTION:
          hasFirmware=BX_FALSE;
          if (task==BX_OPEN_USB23)
          {
            BESTX_STRCPY(buf, "E2923A"); 
            /* BESTX_STRCPY(buf, "E2923F"); */
          }
          else
          {
            BESTX_STRCPY(buf, "E2922A"); 
          }
          break;

#ifdef CUSTOM_OEM1
        case BX_PORT_OEM:
#endif
        case BX_PORT_PCI_CONF:
        case BX_PORT_PCI_IO:
          /* read 4 bytes at address 0x800 (info sect in core) to check
            product string. NOTE: only the first three characters are
            compared, to allow for any combination of E29xx! */
          /* CAUTION: 
            On some systems the following call returns wrong data
            '0000' instead of "E292" and an internal bus error occurs on the 
            DBI bus. As a workaround we call this function twice (second call
            seems to return correct data).
          */
           
          DBG_OUT(("Reading core product string: ",buf));
          BX_TRY (BestXDirectRegBlockRead (handle, 0x800,4, 0, (bx_int8ptr) buf, 1UL, 4));
          BX_TRY (BestXDirectRegBlockRead (handle, 0x800,4, 0, (bx_int8ptr) buf, 1UL, 4));
          buf[3] = '\0';
          DBG_OUT (("\"%s\"\n", buf));

          if (!StrEQ(buf, "E29") || BestXConnect(handle) != BX_E_OK || BestXDisconnect(handle) != BX_E_OK)
          {
            DBG_OUT (("Card has no firmware\n"));
            hasFirmware = BX_FALSE;

            /* use access port for PCI connection */

            /* first, check whether license was programmed into flash */
            BX_TRY (BestXDirectRegRead ( handle,BX_VERSION_SIGNATURE,4UL,&value ));
           
            if (value == BX_VALID_SIGNATURE) /* flash was programmed */
            {
              /* read product string from flash */
              BX_TRY (BestXDirectRegBlockRead( handle,BX_OVERLOADED_PRODUCT_STRING,4UL,0,(bx_int8ptr) buf,1UL,BX_PRODUCT_STRING_LENGTH));
            }
            else
            {
              /* flash was NOT programmed */
              DBG_OUT (("Warning: Missing signature\n", buf));
              BESTX_STRCPY(buf, "E2922A"); /* assume E2922A board */
            }
            break;
          }
          else
          {
            DBG_OUT(("Card has firmware\n"));
          }
          /* no break here intentionally outside "if" clause! */
        default:
          hasFirmware = BX_TRUE;
          /* try to establish a connection, run it down and re-establish it! */
          BX_TRY(BestXConnect(handle));
          BX_TRY(BestXDisconnect(handle));
          BX_TRY(BestXConnect(handle));

#ifdef CUSTOM_OEM1
          if (pHandle->port == BX_PORT_OEM)
          {
            /* set timeouts to default here if PCI port is used!
                (this is needed for the pingquick option) */
            BX_TRY(BestXPortTimeoutSet(handle, NULL, TIMEOUTS_SET_DEF));
          }
#endif

          if (pHandle->port == BX_PORT_PCI_CONF || pHandle->port == BX_PORT_PCI_IO)
          {
            /* set timeouts to default here when PCI port is used!
                (this is needed for the pingquick option) */
            BX_TRY(BestXPortTimeoutSet(handle, NULL, TIMEOUTS_SET_DEF));
          }
  
          if (pHandle->port == BX_PORT_PCI_CONF || pHandle->port == BX_PORT_PCI_IO)
          {
            /* Speedup BestXOpen() for PCI port by reading via access port */    
            /* first, check whether license was programmed into flash */
            BX_TRY (BestXDirectRegRead (handle,BX_VERSION_SIGNATURE,4UL,&value ));

            if (value == BX_VALID_SIGNATURE) /* flash was programmed */
            {
              /* read product string from flash */
              BX_TRY (BestXDirectRegBlockRead( handle,BX_OVERLOADED_PRODUCT_STRING,4UL,0,(bx_int8ptr) buf,1UL,BX_PRODUCT_STRING_LENGTH));
            }
            else
            {
              /* Should never happen */
              /* flash was NOT programmed */
              DBG_OUT (("Warning: Missing signature\n", buf));
              BESTX_STRCPY(buf, "E2922A"); /* assume E2922A board */
            }
          }
          else
          {
            /* ask core for product string, just to make sure! */
            BX_TRY(BestXBasicBlockRead( handle,RPRODUCT_STRING, /* 0x2 */(bx_int8ptr) buf,(bx_int32) BX_PRODUCT_STRING_LENGTH /* 0x14 */));
          }
      }  /* port switch */

      DBG_OUT(("Product string is %s\n",buf));
      buf[BX_PRODUCT_STRING_LENGTH] = '\0'; /* make sure buf is terminated */

      /* load the hardware struct corresponding to the product string */
      for (i = 0; i < NUM_HW_DEFAULTS; i++)
      {
        if (0 == BESTX_STRCMP(buf, hw_info_defs[i].product_string))
        {
          /* good one !! */
          pHandle->hwinfo = hw_info_defs[i];
          break;
        }
      }

      if (i == NUM_HW_DEFAULTS)  /* not found ? */
      {
        /* We dont know this hardware! */
        BestXLastErrorParamSet(handle, BX_ERRPAR_1,(bx_int32) (bx_handlearray[handle].port));
        BX_TRY_ERROR(BX_E_UNKNOWN_HARDWARE);
      }
    }

    /* we found the hardware */

    /* take care of EVERYTHING else that is hardware dependent or make any
     * modifications to bx_handlearray[].hwinfo here... */

    /* we're done if we're working offline */
    if (fIsOffline)
    {
      return BX_E_OK;
    }

    BestXHwBitMaskSet(handle, BX_HWDEF_HAS_FIRMWARE, hasFirmware);

    /* we're done for 22'er Card */
    if(!hasFirmware || task==BX_OPEN_FAST)
    {
      DBG_OUT (("Exiting BestXIdHardware(): no firmware or FastOpen\n"));
      return BX_E_OK;
    }

    /* Check, wether card is in core or bios-mode */
    BX_TRY(BestXBasicBlockRead(handle, VERSION_CODE, (bx_int8ptr) buf, 4UL));

    /* This returns 'cccc' for core or 'bbbb' for bios-mode */

    switch (buf[0])
    {
      case 'c':
        /* core mode */
        BestXLastErrorParamSet(handle, BX_ERRPAR_1,(bx_int32) (bx_handlearray[handle].port));
        BX_TRY_FAIL(task == BX_OPEN_BIOS_SWITCH ?BX_E_OK : BX_E_CANNOT_CONNECT_CORE);
        pHandle->hwinfo.hwbitmask &= ~BX_HWDEF_HAS_FHIF;
        pHandle->hwinfo.hwbitmask |= BX_HWDEF_IS_CORE;
        break;

      case 'b':
        break;

      default:
        BX_TRY_FAIL(BX_E_INVALID_CASE); /* should never happen */
        break;
    }

    if (fDoVerCheck) /* false for BIOS_SWITCH */
    {
      /* We are in BIOS mode here ! */
      bx_int32 BiosVersionNumberFromCard,BiosVersionNumberFromCapi;
      bx_vertype Bios1XVersionFromCapi=BX_VER_INIT(BX_VER_BIOS); 
      bx_vertype Bios2XVersionFromCapi=BX_VER_INIT(BX_VER_BIOS2X); 
      bx_vertype BiosVersionFromCapi;

      /* chris PCIX2.0 devices have separate BIOS version ! */
      if ((bx_handlearray[handle].hwinfo.hwseries==BX_SERIES_E2930A) ||
          (bx_handlearray[handle].hwinfo.hwseries==BX_SERIES_E2930B)
         )
      {
        BiosVersionFromCapi=Bios2XVersionFromCapi;
      }
      else
      {
        BiosVersionFromCapi=Bios1XVersionFromCapi;
      }

      /* Get bios version number from CAPI */
      BiosVersionNumberFromCapi=BX_VER_NUMBER(&BiosVersionFromCapi);

      /* Read bios version number from card */

      /* get version number from bios and check that FW==CAPI versions */
      /* This is a string of the form: "0x12345678" */
      BX_TRY(BestXBasicBlockRead(handle, BIOS_VERSION_NUMBER, (bx_int8ptr) buf, 11UL));
      BiosVersionNumberFromCard=BESTX_STRTOUL(buf,NULL,16);

#if 0 /* TDB: chris error parameters for version mismatch error */
      BESTX_STRCPY(s);
      BESTX_STRCAT(s, "@");
      BESTX_STRCAT(s, buf);
      BestXLastErrorParamStringSet(handle, s);
      BestXLastErrorParamSet(handle, BX_ERRPAR_1, (bx_int32) BX_VERCHK_FIRMWARE);
#endif

      /* We ignore the patch number (low-byte) here ! */
      if ((BiosVersionNumberFromCard & 0xffffff00) != (BiosVersionNumberFromCapi & 0xffffff00))
      {
        /* Patch-part is ignored here ! */
        BestXLastErrorParamSet(handle, BX_ERRPAR_1,BiosVersionNumberFromCapi);
        BestXLastErrorParamSet(handle, BX_ERRPAR_2,BiosVersionNumberFromCard );
        BX_TRY_FAIL(BX_E_VERSION_MISMATCH);
      }
    }
    (void) BestXDisconnect(handle);
  }

  BX_ERRETURN(BX_TRY_RET);
}


/* --------------------------------------------------------------------------
* close the connection to the board and throw away all information
* -------------------------------------------------------------------------- */

bx_errtype EXPORT BestXClose(bx_handletype handle)
{
  BX_ERRETURN(BestXCloseAdv(handle, BX_CLOSE_STANDARD));
}


bx_errtype BestXCloseAdv(bx_handletype handle, bx_closetype task)
{
  BX_TRY_VARS_NO_PROG;
  bx_porttype port;
  bx_portnumtype os_handle;
  BX_TRY_BEGIN
  {
    BX_TRY_HANDLE;
    BX_TRY(BestXCapaClose(handle)); /* delete the dynamic capabilities */
    BX_TRY(BestXDBClose(handle)); /* delete database */
    DataBufferDelete(handle);   /* get rid of general purpose buffer */
    (void) BestXGetPortFromHandle(handle, &port);

    /* save os handle for later use... */
    os_handle = bx_handlearray[handle].portnumber;

    /* unconditional disconnect */
    BestXReleaseConnection(handle);

    /* Total reset...this must be the last thing we do here. Note that this
    * is the only place that we "disconnect" a 2926 and is therefore
    * essential */

#ifndef _AIX
    bx_handlearray[handle] = xhandle_null;

    /* well, except the reserved field */
    bx_handlearray[handle].is_reserved = (bx_int8) (task == BX_CLOSE_RESERVED ? 1 : 0);
#endif

    /* the handle is now officially invalid */
    switch (port)
    {
      case BX_PORT_OFFLINE:
        break;

#if !defined(AIX) && !defined(_AIX)
      case BX_PORT_RS232:
        BestXCloseSerial(os_handle);
        break;

      case BX_PORT_PARALLEL:
        BestXCloseParallel(os_handle);
        break;

      case BX_PORT_FASTHIF:
        (void) BestXCloseHIF(os_handle);
        break;

      case BX_PORT_PRODUCTION:
        if (task!=BX_CLOSE_USB23)
        {
          break;
        }
        else
        {
          /* USB23 production needs to close USB handle */
        }
        /* No break here intentionally ! */
      case BX_PORT_USB:
        (void) BestXCloseUSB(os_handle);
        break;
#endif

#ifdef CUSTOM_OEM1
      case BX_PORT_OEM:
        (void) BestXClosePCI( os_handle ) ;
        (void) BestXCloseOEM( bx_handlearray[handle].OEMData ) ;
        break ;
#endif


      case BX_PORT_PCI_CONF:
      case BX_PORT_PCI_IO:
        (void) BestXClosePCI(os_handle);
        break;

      default:
        break;
    }     
#if _AIX
  /* We null out the handle below instead of above because BestXClosePCI
   * still needs to have access to the entries within the struct  */
    bx_handlearray[handle] = xhandle_null;

    /* well, except the reserved field */
    bx_handlearray[handle].is_reserved = (bx_int8) (task == BX_CLOSE_RESERVED ? 1 : 0);
#endif
  
  }

  BX_ERRETURN(BX_TRY_RET);
}


/* -------------------------------------------------------------------------
* Set the baudrate of the RS232 Interface to value baudrate
* ------------------------------------------------------------------------- */
bx_errtype EXPORT BestXRS232BaudRateSet(
                                        bx_handletype handle,
                                        bx_int32 baudrate
                                        )
{
#if !defined(AIX) && !defined(_AIX)
  BX_TRY_VARS_NO_PROG;
  bx_portnumtype OsHandle = INVALID_OS_HANDLE;
  int exclusive;
  BX_TRY_BEGIN
  {
    BX_TRY_FAIL(bx_handlearray[handle].port == BX_PORT_RS232 ?
                                             BX_E_OK : BX_E_WRONG_PORT);
    BX_TRY_FAIL(BestXIsHandleConnected(handle) ? BX_E_OK : BX_E_NOT_CONNECTED);

    exclusive = (bx_handlearray[handle].is_connected == CONNECT_EXCLUSIVE);

    if (!BestXIsCore(handle))
    {
      bx_int8 inbuf[4];
      (void) BestXLong2Stream(inbuf, &baudrate, 1UL);
      BX_TRY(BestXBasicCommand(handle, CMD_SER_BAUDRATE_SET,
        inbuf, IN_SER_BAUDRATE_SET,
        NULL, NULL));
    }
    else                        /* core mode */
    {
      /* set new baudrate in Core mode */
      /* TODO: in error returned BX_E_BAUDRATE */
      BX_TRY(BestXBasicBlockWrite(handle,
          RS232_BAUDRATE, (bx_int8ptr) & baudrate, 4UL));
    }

    /* now set new baudrate on host */
    bx_handlearray[handle].param = baudrate;  /* for later queries only */

    /* change COM port settings to new baudrate */
    BX_TRY(BestXSerInit(bx_handlearray[handle].portnumber, baudrate));

    /* release connection, just to make sure */
    BestXSerReleaseConnection (bx_handlearray[handle].portnumber);

    /* wait for card to release connection */
    BX_TRY (BestXWaitForClose (handle, 1000));

    /* establish new connection */
    BX_TRY(BestXOpenConnection(handle, xconnect_timeout_array[handle]));
  }

  return BX_TRY_RET;
#else
  return BX_E_INVALID_CASE;
#endif
}


/***********************************************************************
  handle related functions (not in firmware)
***********************************************************************/
bx_bool BestXIsOffline(bx_handletype handle)
{
  return (bx_bool) (__BX_HANDLECHECK ?
    bx_handlearray[handle].port==BX_PORT_OFFLINE : 0);
}

bx_errtype EXPORT BestXDebug(bx_handletype handle, bx_int32 debug)
{
  BX_DECLARE_FUNCNAME("BestXDebug");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    BX_TRY_FCT_PARAM(1, debug > 0x1);
    BX_TRY_HANDLE;

    if (debug != 0)
    {
      bx_handlearray[handle].cbx_printf = bx_handlearray[handle].cbx_p_backup;
    }
    else
    {
      bx_handlearray[handle].cbx_printf = NULL;
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}


bx_errtype EXPORT BestXGetDebugOutFromHandle(bx_handletype handle,
                                             void (**std_out) (char *,...))
{
  BX_HANDLECHECK;
  *std_out = bx_handlearray[handle].cbx_printf;
  BX_ERRETURN(BX_E_OK);
}


bx_errtype EXPORT BestXGetPortFromHandle(bx_handletype handle,
                                         bx_porttype * port)
{
  BX_HANDLECHECK;
  *port = bx_handlearray[handle].port;
  BX_ERRETURN(BX_E_OK);
}


bx_errtype EXPORT BestXGetPortnumberFromHandle(bx_handletype handle,
                                               bx_int32 * portnumber)
{
  BX_HANDLECHECK;
  *portnumber = bx_handlearray[handle].entered_port;
  BX_ERRETURN(BX_E_OK);
}




bx_errtype EXPORT BestXGetHWFromHandle(bx_handletype handle,
                                       bx_hwtype * hw)
{
  BX_HANDLECHECK
  * hw = bx_handlearray[handle].hwinfo.hw;
  BX_ERRETURN(BX_E_OK);
}

bx_errtype EXPORT BestXGetHWSeriesFromHandle(bx_handletype handle,
                                             bx_hwseriestype *hwseries)
{
  BX_HANDLECHECK
  *hwseries = bx_handlearray[handle].hwinfo.hwseries;
  BX_ERRETURN(BX_E_OK);
}

bx_errtype EXPORT BestXGetPciDevIdFromHandle(bx_handletype handle,
                                             bx_int32 * dev_id)
{
  BX_HANDLECHECK
  * dev_id = bx_handlearray[handle].hwinfo.dev_id;
  BX_ERRETURN(BX_E_OK);
}

bx_errtype EXPORT BestXGetProductstringFromHandle(bx_handletype handle,
                                                bx_charptrtype * pChar)
{
  BX_HANDLECHECK;
  assert (pChar != NULL);
  *pChar = bx_handlearray[handle].hwinfo.product_string;
  BX_ERRETURN(BX_E_OK);
}

/* --------------------------------------------------------------------------
* HARDWARE CHARACTERISTICS
* The following functions are used to distinguish hardware.
* Because these functions are often called before any handle checks it
* is possible that we have a bad handle but we don't have a bx_errtype to
* return....therefore we return the value which is most likely to make
* the calling function continue and call a routine which will trap the
* handle error.
* -------------------------------------------------------------------------- */

bx_bool EXPORT BestXIsOpen(bx_handletype handle)
{
  /* HL; return FALSE (!!!!) if this is a bad handle */
  return (bx_bool) (__BX_HANDLECHECK ? 1 : 0 );
}

bx_bool EXPORT BestXIsKnown(bx_handletype handle)
{
  /* CZ; return FALSE (!!!!) if this is a bad handle */
  return (bx_bool) (__BX_HANDLECHECK ?
    bx_handlearray[handle].hwinfo.hwseries != BX_SERIES_UNKNOWN : 0);
}

bx_bool EXPORT BestXHasFastHost(bx_handletype handle)
{
  /* SCR; return FALSE if this is a bad handle */
  return (bx_bool) (__BX_HANDLECHECK ?
    (bx_handlearray[handle].hwinfo.hwbitmask & BX_HWDEF_HAS_FHIF) != 0 : 0);
}

bx_bool EXPORT BestXHasFaust(bx_handletype handle)
{
  /* SCR; return FALSE if this is a bad handle */
  return (bx_bool) (__BX_HANDLECHECK ?
    (bx_handlearray[handle].hwinfo.hwbitmask & BX_HWDEF_HAS_FAUST) != 0 : 0);
}

bx_bool EXPORT BestXHasFaustFPGA(bx_handletype handle)
{
  return (bx_bool) (__BX_HANDLECHECK ?
    (bx_handlearray[handle].hwinfo.hwbitmask & BX_HWDEF_HAS_FAUSTFPGA) != 0 : 0);
}

bx_bool EXPORT BestXHasMephisto(bx_handletype handle)
{
  /* SCR; return FALSE if this is a bad handle */
  return (bx_bool) (__BX_HANDLECHECK ?
    (bx_handlearray[handle].hwinfo.hwbitmask & BX_HWDEF_HAS_MEPHISTO) != 0 : 0);
}

bx_bool EXPORT BestXIsMode2(bx_handletype handle)
{
  /* returns true, if initialization pattern contains PERR# asserted (mode2) */

  bx_int32 resetcode;
  bx_errtype err;

  if (!__BX_HANDLECHECK) return 0;
  if (BestXHasMephisto(handle)) return 0;
  if (BestXFaustLoadError(handle)) return 0;

  err=BestXDirectRegRead(handle,BX_REG_SYS_RESETPATT_F,sizeof(bx_int32),&resetcode);
  if (err) return 0;
  resetcode = resetcode & 0xf; /* Extract bits 0-3 */

  if (!(resetcode & (1<<3)))
  {
    return (bx_bool) 1;
  }
  else
  {
    return (bx_bool) 0;
  }
}

bx_bool EXPORT BestXIsQDR(bx_handletype handle)
{
  /* returns true, if initialization pattern contains PERR# and DEVSEL# asserted (mode2 533) */

  bx_int32 resetcode;
  bx_errtype err;

  if (!BestXIsMode2(handle)) return 0;

  err=BestXDirectRegRead(handle,BX_REG_SYS_RESETPATT_F,sizeof(bx_int32),&resetcode);
  if (err) return 0;
  resetcode = resetcode & 0xf; /* Extract bits 0-3 */

  if (!(resetcode & (1<<0)))
  {
    return (bx_bool) 1;  
  }
  else
  {
    return (bx_bool) 0;  
  }
}

bx_bool EXPORT BestXFaustLoadError(bx_handletype handle)
{
  /* Returns TRUE, if:
     - FPGA hardware (E2930A) AND
     - not OFFLINE AND
     - Faust FPGA image could not be loaded
 */

  return (bx_bool) (__BX_HANDLECHECK ?
    (bx_handlearray[handle].FaustLoadError):1);
}

bx_bool EXPORT BestXHasPerformance(bx_handletype handle)
{
  /* SCR; return FALSE if this is a bad handle */
  return (bx_bool) (__BX_HANDLECHECK ?
    (bx_handlearray[handle].hwinfo.hwbitmask & BX_HWDEF_HAS_PERF) != 0 : 0);
}

bx_bool EXPORT BestXHasFirmware(bx_handletype handle)
{
  /* SCR; return FALSE if this is a bad handle */
  return (bx_bool) (__BX_HANDLECHECK ?
    (bx_handlearray[handle].hwinfo.hwbitmask & BX_HWDEF_HAS_FIRMWARE) != 0 : 0);
}

bx_bool EXPORT BestXIsDeep(bx_handletype handle)
{
  /* SCR; return FALSE if this is a bad handle */
  return (bx_bool) (__BX_HANDLECHECK ?
    (bx_handlearray[handle].hwinfo.hwbitmask & BX_HWDEF_IS_DEEP) != 0 : 0);
}

bx_bool EXPORT BestXIsCompact(bx_handletype handle)
{
  return (bx_bool) (__BX_HANDLECHECK ?
    (bx_handlearray[handle].hwinfo.hwbitmask & BX_HWDEF_IS_COMPACT) != 0 : 0);
}

bx_bool EXPORT BestXIsCore(bx_handletype handle)
{
  return (bx_bool) (__BX_HANDLECHECK ?
    (bx_handlearray[handle].hwinfo.hwbitmask & BX_HWDEF_IS_CORE) != 0 : 0);
}


bx_bool EXPORT BestXIsInBlockMode(bx_handletype handle)
{
  return (bx_bool) (__BX_HANDLECHECK ?
    (bx_handlearray[handle].hwinfo.hwbitmask & BX_HWDEF_IS_BLKMODE) != 0 : 0);
}

bx_bool EXPORT BestXIsInDirectMode(bx_handletype handle)
{
  return (bx_bool) (__BX_HANDLECHECK ?
    (bx_handlearray[handle].hwinfo.hwbitmask & BX_HWDEF_IS_DIRECTMODE) != 0 : 0);
}

void BestXHwBlockModeBitSet(bx_handletype handle, bx_bool fSet)
{
  BestXHwBitMaskSet(handle, BX_HWDEF_IS_BLKMODE, fSet);
}

void BestXHwDirectModeBitSet(bx_handletype handle, bx_bool fSet)
{
  BestXHwBitMaskSet(handle, BX_HWDEF_IS_DIRECTMODE, fSet);
}

void BestXHwBitMaskSet(bx_handletype handle, bx_int32 mask, bx_bool fSet)
{
  if(__BX_HANDLECHECK)
  {
    if(fSet)
    {
      bx_handlearray[handle].hwinfo.hwbitmask |= mask;
    }
    else
    {
      bx_handlearray[handle].hwinfo.hwbitmask &= ~mask;
    }
  }

  return;
}

bx_bool BestX16BitRegisterFile(bx_handletype handle)
{
  /* 8 bit register file only used in core mode (CS) */
  /* SCR; return false if this is a bad handle */
  return (bx_bool) (__BX_HANDLECHECK ?
    BestXIsCore(handle)==0 : 0);
}

bx_hwseriestype EXPORT BestXHwSeriesGet(bx_handletype handle)
{
  return (__BX_HANDLECHECK ?
    bx_handlearray[handle].hwinfo.hwseries : BX_SERIES_UNKNOWN);
}

/* --------------------------------------------------------------------------
* SCR; 29.10.97; Added the new I/O scheme...see iocommon.c
* Note; 2926's must still call this AT LEAST ONCE during an Open.
* -------------------------------------------------------------------------- */

bx_errtype EXPORT BestXConnect(bx_handletype handle)
{
  BX_TRY_VARS_NO_PROG;

  BX_HANDLECHECK;

  BX_TRY_BEGIN
  {
    /* real connects are only done for the 2925 */
    if (!BestXIsKnown(handle))
    {
      BX_TRY(BestXOpenConnection(handle, xconnect_timeout_array[handle]));
    }

    bx_handlearray[handle].is_connected = CONNECT_NORMAL;
  }

  BX_TRY_CATCH
  {
    /* SCR; 11.97; BestXReleaseConnection() now waits for close */
    BestXReleaseConnection(handle);
  }

  BX_ERRETURN(BX_TRY_RET);
}



/* --------------------------------------------------------------------------
* TODO; This may be an obsolete function....where used?
* -------------------------------------------------------------------------- */

bx_errtype EXPORT BestXConnectEx(bx_handletype handle)
{
  BX_TRY_VARS;
  bx_int8 zw;
  volatile int was_connected = 0;
  /* TODO ... this was done to allow old code and OFFLINE to work ! What do
  * we really want to do with 16BitRegisterFile ? */

  BX_TRY_BEGIN
  {
    if (!BestXIsKnown(handle))
    {
      was_connected = BestXIsHandleConnected(handle);
      BX_TRY_PROGRESS(BestXConnect(handle));

      if (BX_PORT_OFFLINE != bx_handlearray[handle].port)
      {
        if(!BestXHasFirmware(handle))
        {
          BX_TRY_FAIL(BX_E_ERROR);
        }
        zw = LOCK_BOARD;
        BX_TRY(BestXBasicBlockWrite(handle, SERVICE_CMD, &zw, 1UL));
      }

      bx_handlearray[handle].is_connected = CONNECT_EXCLUSIVE;
    }
  }

  BX_TRY_CATCH
  {
    BX_TRY_RET = BX_E_CANNOT_CONNECT_EXCLUSIVE;
    BestXLastErrorParamSet(handle, BX_ERRPAR_1,
      (bx_int32) (bx_handlearray[handle].port));

    BX_TRY_PASSED
    {
      /* nothing to do */
    }

    BX_TRY_FAILED
    {
      was_connected = 0;
    }

    if (!was_connected)
    {
      (void) BestXDisconnect(handle);
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}



/* --------------------------------------------------------------------------
* SCR; Note that a 2926 will NEVER DISCONNECT until closed.
* Also note that we check for an open handle (even in 2926 mode) just to
* preserve the logic.
* -------------------------------------------------------------------------- */

bx_errtype EXPORT BestXDisconnect(bx_handletype handle)
{
  /* here for debugging purposes on the 2926 */
  BX_HANDLECHECK;

  if (!BestXIsKnown(handle))
  {
    /* chris: BestXReleaseConnection() waits for disconnection acknowledge from card */
    BestXReleaseConnection(handle);
    bx_handlearray[handle].is_connected = 0;
  }

  BX_ERRETURN(BX_E_OK);
}


bx_errtype EXPORT BestXDummySet(bx_handletype handle, bx_int32 dummy)
{
  if(!BestXHasFirmware(handle))
  {
    BX_ERRETURN(BX_E_OK);
  }
  BX_ERRETURN(BestXBasicBlockWrite(handle, DUMMY_REGISTER, (bx_int8ptr) & dummy, 4UL));
}


bx_errtype EXPORT BestXDummyGet(bx_handletype handle, bx_int32 * dummy)
{
  if(!BestXHasFirmware(handle))
  {
    BX_ERRETURN(BX_E_OK);
  }
  BX_ERRETURN(BestXBasicBlockRead(handle, DUMMY_REGISTER, (bx_int8ptr) dummy, 4UL));
}


bx_errtype EXPORT BestXPing(bx_handletype handle)
{
  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    if(BestXHasFirmware(handle))
    {
      BX_TRY(BestXBasicCommand(handle, CMD_PING, NULL, IN_PING, NULL, NULL));
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}


bx_errtype EXPORT BestXClockSet (bx_handletype handle, bx_int32 clkvalue)
{
  bx_errtype    err;

  if(!BestXHasFirmware(handle))
  {
    BX_ERRETURN(BX_E_OK);
  }
  err = BestXBBCSendLong (handle, CMD_CLOCK_SET, &clkvalue);
  BX_ERRETURN (err);
}


bx_errtype EXPORT BestXClockGet (bx_handletype handle, bx_int32 *clkvalue)
{
  bx_errtype    err;

  if(!BestXHasFirmware(handle))
  {
    BX_ERRETURN(BX_E_OK);
  }

  err = BestXBBCReceiveLong (handle, CMD_CLOCK_GET, clkvalue);
  BX_ERRETURN (err);
}

#endif /* firmware */



/***************************************************************************
* Locking functions
* Resources ( Exerciser, Analyzer, Observer, Performance ) can be locked by
* ports (PCI or DBI (FastHost/RS232/USB)). In case the OFFLINE port is used,
* the exported functions:
* {BestXResourceLock, BestXResourceUnlock, BestXAllResourceUnlock}
* return BX_E_OK.
**************************************************************************/

static bx_errtype BestXResourceUnlockFunction(bx_handletype handle,
                                              bx_int32 Port_Mask,
                                              bx_resourcetype resource)
{
  BX_TRY_VARS_NO_PROG;
  bx_int32 ReslockReg=0;

  BX_TRY_BEGIN
  {
        
    /* Read value of the resource register */
    BX_TRY(BestXDirectRegRead(handle,
                            BX_REG_RES_LOCK_REG,
                            sizeof(bx_int32),
                            &ReslockReg
                            ));

    ReslockReg^=Port_Mask; /* we remove setting ...*/
    BX_TRY(BestXDirectRegWrite (handle,
                              BX_REG_RES_LOCK_REG,
                              sizeof(bx_int32),
                              ReslockReg
                              ));

    /*REMOVE:*/
    /* printf("And unlocking ...\n");      */
    BestXResourceLockDecreaseCounter(handle, resource);

  }/* BX_TRY_BEGIN */
  BX_ERRETURN(BX_TRY_RET);
} /* BestXResourceUnlockFunction */


/*---------------------------------------------------------------------
* Helping function to make the code more readable:
* Just increases value of counter for resource. The counter is increased
* every time a port locks a resource.
*---------------------------------------------------------------------*/

static bx_errtype BestXResourceLockIncreaseCounter(bx_handletype handle,
                                                   bx_resourcetype resource)
{
  BX_TRY_VARS_NO_PROG;
  BX_TRY_BEGIN
  {
    switch(resource)
    {
    case BX_RESLOCK_EXERCISER:
      BX_EXERCISER_RESLOCK_CTR++;
      break;
    case BX_RESLOCK_OBSERVER:
      BX_OBSERVER_RESLOCK_CTR++;
      break;
    case BX_RESLOCK_PERFORMANCE:
      BX_PERFORMANCE_RESLOCK_CTR++;
      break;
    case BX_RESLOCK_ANALYZER:
      BX_ANALYZER_RESLOCK_CTR++;
      break;
    default:
      BX_TRY_ERROR(BX_E_INVALID_CASE);
      break;
    } /*switch(resource)*/
  } /* BX_TRY_BEGIN */
  BX_ERRETURN(BX_TRY_RET);
} /* BestXResourceLockIncreaseCounter */

/*---------------------------------------------------------------------
* Helping function to make the code more readable:
* Just decreases value of counter for resource. The counter is decreased
* every time a port unlocks a resource.
*-----------------------------------------------------------------------*/

static bx_errtype BestXResourceLockDecreaseCounter(bx_handletype handle,
                                                   bx_resourcetype resource)
{
  BX_TRY_VARS_NO_PROG;
  BX_TRY_BEGIN
  {
    switch(resource)
    {
      case BX_RESLOCK_EXERCISER:
        BX_EXERCISER_RESLOCK_CTR--;
        break;
      case BX_RESLOCK_OBSERVER:
        BX_OBSERVER_RESLOCK_CTR--;
        break;
      case BX_RESLOCK_PERFORMANCE:
        BX_PERFORMANCE_RESLOCK_CTR--;
        break;
      case BX_RESLOCK_ANALYZER:
        BX_ANALYZER_RESLOCK_CTR--;
        break;
      default:
        BX_TRY_ERROR(BX_E_INVALID_CASE);
        break;
    } /*switch(resource)*/
  } /* BX_TRY_BEGIN */
  BX_ERRETURN(BX_TRY_RET);
} /* BestXResourceLockIncreaseCounter */

/*******************************************************************
* This is the actual function that locks the resource. The procedure
* is as follows:
* 1. Read the value of the register on the card
* 2. Check if other port already locked the resource
*    (there are four kinds of resources: Analyzer,Exerciser,Observer
*     and Performance)
*    (there are two kind of ports: PCI or RS232/FASTHOST/USB)
* 3. If other port has already locked the resource -> error msg
* 4. If not, modify value and write value to card
* 5. Now read value again from card to make sure no other port locked
*    the resource in the meantime.
* 6. If another port locked the resource -> undo (4.) and -> error msg
* 7. If not, increase Counter.
*********************************************************************/

static bx_errtype BestXResourceLockFunction(bx_handletype handle,
                                           bx_int32 This_Port_Mask,
                                           bx_int32 Other_Port_Mask,
                                           bx_resourcetype resource)
{
bx_int32 ReslockReg;
BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
   /* Read value of the resource register */
    BX_TRY(BestXDirectRegRead(handle,
                            BX_REG_RES_LOCK_REG,
                            sizeof(bx_int32),
                            &ReslockReg
                            ));

    /* Check if other port locked it already */
    if((Other_Port_Mask & ReslockReg)!=0)
    {
      BX_ERRETURN(BX_E_RESOURCE_LOCKED); /* already locked by another port ! */
    }
    else    /* not locked by another port, let's lock ! */
    {
      ReslockReg|=This_Port_Mask; /* Modify register with lock-bit */
      BX_TRY(BestXDirectRegWrite (handle,
                              BX_REG_RES_LOCK_REG,
                              sizeof(bx_int32),
                              ReslockReg
                              ));
      /* read again if in meantime another port locked resource */
      BX_TRY(BestXDirectRegRead (handle,
                              BX_REG_RES_LOCK_REG,
                              sizeof(bx_int32),
                              &ReslockReg
                              ));

      if((Other_Port_Mask & ReslockReg)!=0)
      {/* it actually happend, another port locked the resource ! */
        ReslockReg^=This_Port_Mask; /* we undo ...*/
        BX_TRY(BestXDirectRegWrite (handle,
                              BX_REG_RES_LOCK_REG,
                              sizeof(bx_int32),
                              ReslockReg
                              ));
        BX_ERRETURN(BX_E_RESOURCE_LOCKED); /*already locked by another device*/
      } /* if other port locked in meantime */
      else /* we succeeded in locking !*/
      {
        /*REMOVE: */
        /* printf("We are locking ...\n"); */
        BestXResourceLockIncreaseCounter(handle, resource);

      } /* else we succeded ... */
    }/* if locked by another port */
  } /*BX_TRY_BEGIN */
BX_ERRETURN(BX_TRY_RET);
} /* BestXResourceLockFunction */


/*-----------------------------------------------------------------
* Helping function to make code more readable:
*  Determines if port is PCI or other (RS232/FastHost/USB etc.)
------------------------------------------------------------------*/

static bx_errtype BestXResourceLockDeterminePort(bx_handletype handle,
                                                 bx_int32 *current_port)
{
  BX_TRY_VARS_NO_PROG;
  bx_porttype port;
  BX_TRY_BEGIN
  {
   BX_TRY(BestXGetPortFromHandle(handle,&port));

    switch(port)
    {
#ifdef CUSTOM_OEM1
      case BX_PORT_OEM:
        *current_port=BX_PORT_IS_PCI;
        break;
#endif
      case BX_PORT_PCI_CONF:
        *current_port=BX_PORT_IS_PCI;
        break;
      case BX_PORT_PCI_IO:
        *current_port=BX_PORT_IS_PCI;
        break;
      case BX_PORT_OFFLINE:
        *current_port=BX_PORT_IS_OFFLINE; /* no resourcelocking for offline ..*/
        break;
      case BX_PORT_USB:
        *current_port=BX_PORT_IS_USB;
        break;
      case BX_PORT_RS232:
        *current_port=BX_PORT_IS_RS232;
        break;
      case BX_PORT_FASTHIF:
        *current_port=BX_PORT_IS_FHIF;
        break;
      case BX_PORT_PRODUCTION:
        *current_port=BX_PORT_IS_PRODUCTION;
        break;
      default:
      BX_TRY_ERROR(BX_E_INVALID_CASE);
        break;
     } /* switch (port) */
  } /*BX_TRY_BEGIN*/
BX_ERRETURN(BX_TRY_RET);
}/* BestXResourceLockDeterminePort */


/*-------------------------------------------------------------------
* This function locks a resource.
* As follows:
* 1. Check if what kind of port is used (PCI) or (FASTHOST/RS232/USB)
*    or OFFLINE. In latter case just return BX_E_OK
* 2. Using function 'BestXResourceIsLocked we check, if only the counter
*    on the host PC has to be increased, or if the value has to be
*    written on the register on the card. In this case, the function
*    'BestXResourceLockFunction' is called with the values of the
*    masks for the current port.
-----------------------------------------------------*/

bx_errtype EXPORT BestXResourceLock(
  bx_handletype handle,
  bx_resourcetype resource)
{
  BX_DECLARE_FUNCNAME("BestXResourceLock [reslock]");
  BX_TRY_VARS_NO_PROG;
  bx_int32 current_port;       /* Port in use right now (PCI or DBI)*/
  bx_porttype port;
  bx_int32ptr valptr=NULL;   /* value of ReslockReg */
  bx_int32 lockcount=0;      /* number of times the port was locked. 0=not locked */
  bx_int32 This_Port_Mask;
  bx_int32 Other_Port_Mask;
  bx_int32 i;
  bx_int32 Exerciser_Masks;
  bx_int32 Analyzer_Masks;
  bx_int32 Observer_Masks;
  bx_int32 Performance_Masks;

  BX_HANDLECHECK;

  BX_TRY_BEGIN
  {

    BX_TRY(BestXResourceLockDeterminePort(handle,&current_port));
    if( (current_port==BX_PORT_IS_OFFLINE) || (current_port==BX_PORT_IS_PRODUCTION))
       BX_ERRETURN(BX_E_OK);

    /* check if the resource is already locked and by which port. */
    /* store result of port in 'port' and number of lockings in 'lockcount'*/
    BX_TRY(BestXResourceIsLocked(handle,resource,&lockcount,&port));

    
    /* build masks, one resource (eg exerciser) for all ports. */
    i=BX_EXERCISER_LOCK_MASK;
    Exerciser_Masks=((i << BX_PORT_IS_RS232 ) | ( i << BX_PORT_IS_PCI ) |
                     (i << BX_PORT_IS_FHIF  ) | ( i << BX_PORT_IS_USB ));

    i=BX_ANALYZER_LOCK_MASK;
    Analyzer_Masks=((i << BX_PORT_IS_RS232 ) | ( i << BX_PORT_IS_PCI ) |
                     (i << BX_PORT_IS_FHIF  ) | ( i << BX_PORT_IS_USB ));

    i=BX_OBSERVER_LOCK_MASK;
    Observer_Masks=((i << BX_PORT_IS_RS232 ) | ( i << BX_PORT_IS_PCI ) |
                     (i << BX_PORT_IS_FHIF  ) | ( i << BX_PORT_IS_USB ));

    i=BX_PERFORMANCE_LOCK_MASK;
    Performance_Masks=((i << BX_PORT_IS_RS232 ) | ( i << BX_PORT_IS_PCI ) |
                     (i << BX_PORT_IS_FHIF  ) | ( i << BX_PORT_IS_USB ));


    if(lockcount==0) /* not yet locked on card */
    {
      switch(resource)
      {
        case BX_RESLOCK_EXERCISER:
          This_Port_Mask=(BX_EXERCISER_LOCK_MASK<<current_port);
          Other_Port_Mask=(Exerciser_Masks^This_Port_Mask);
          break;
        case BX_RESLOCK_OBSERVER:
          This_Port_Mask=(BX_OBSERVER_LOCK_MASK<<current_port);
          Other_Port_Mask=(Observer_Masks^This_Port_Mask);
          break;
        case BX_RESLOCK_PERFORMANCE:
          This_Port_Mask=(BX_PERFORMANCE_LOCK_MASK<<current_port);
          Other_Port_Mask=(Performance_Masks^This_Port_Mask);
          break;
        case BX_RESLOCK_ANALYZER:
          This_Port_Mask=(BX_ANALYZER_LOCK_MASK<<current_port);
          Other_Port_Mask=(Analyzer_Masks^This_Port_Mask);
          break;
        default:
          BX_TRY_ERROR(BX_E_INVALID_CASE);
          break;
      } /*switch(resource)*/

    BX_TRY(BestXResourceLockFunction(handle, This_Port_Mask, Other_Port_Mask,resource));

    }
    else /*if(lockcount==0)*/
    {
      BestXResourceLockIncreaseCounter(handle, resource);
    } /*if(lockcount==0)*/

  } /* BX_TRY_BEGIN */

  BX_ERRETURN(BX_TRY_RET);

} /*BestXResourceLock */


/* --------------------------------------------------------------------------
* Unlock a specific resource
* As follows:
* 1. Check if port is PCI or (FastHost/RS232/USB). If port is OFFLINE
*    just return BX_E_OK
* 2. Using 'BestXResourceIsLocked' we check if Unlock has only to reduce
*    the counter (on host PC) or has to update register on card
* 3. In former case call 'ResourceLockDecreaseCounter', in latter case
*    call 'BestXResourceUnlockFunction'. For this case, choose the correct
*    Mask for the current port.
* ------------------------------------------------------------------------- */

bx_errtype EXPORT BestXResourceUnlock(
  bx_handletype handle,
  bx_resourcetype resource)
{
  BX_DECLARE_FUNCNAME("BestXResourceUnlock [resunlock]");
  BX_TRY_VARS_NO_PROG;
  bx_int32 current_port=0; /* port right now */
  bx_porttype port;
  bx_int32 lockcount;
  bx_int32 Port_Mask;

  BX_HANDLECHECK;

  BX_TRY_BEGIN
  {

   BX_TRY(BestXResourceLockDeterminePort(handle,&current_port));
   if( (current_port==BX_PORT_IS_OFFLINE) || (current_port==BX_PORT_IS_PRODUCTION))
       BX_ERRETURN(BX_E_OK);

   BX_TRY(BestXResourceIsLocked(handle,resource,&lockcount,&port));

   if(port != BX_PORT_CURRENT)
   {
     BX_ERRETURN(BX_E_RESOURCE_LOCKED); /* already locked by another port ! */
   }
 
    if(lockcount>0)
    {
      if(lockcount==1)
      {
        switch(resource)
        {
          case BX_RESLOCK_EXERCISER:
            Port_Mask=BX_EXERCISER_LOCK_MASK << current_port;
            break;
          case BX_RESLOCK_OBSERVER:
            Port_Mask=BX_OBSERVER_LOCK_MASK << current_port;
            break;
          case BX_RESLOCK_PERFORMANCE:
            Port_Mask=BX_PERFORMANCE_LOCK_MASK << current_port;
            break;
          case BX_RESLOCK_ANALYZER:
            Port_Mask=BX_ANALYZER_LOCK_MASK << current_port;
            break;
          default:
            BX_TRY_ERROR(BX_E_INVALID_CASE);
            break;
        } /*switch(resource)*/

        BX_TRY(BestXResourceUnlockFunction(handle, Port_Mask,resource));
      } /* if(lockcount==1) */
        else
      {
        BestXResourceLockDecreaseCounter(handle, resource);
      } /* else ..if(lockcount==1) */

    } /* if (lockcount>0) */

} /*BX_TRY_BEGIN */

  BX_ERRETURN(BX_TRY_RET);

} /* BestXResourceUnlock */


/* -------------------------------------------------------------------------
* unlock all resources immediately (this is for the case that someone forgot
* to unlock a resource and you need to talk to the card)
* The resourcecounters are set to zero and the bits in the register on the
* card that store the locking information are zeroed
* -------------------------------------------------------------------------*/
bx_errtype EXPORT BestXAllResourceUnlock(bx_handletype handle)
{
  BX_DECLARE_FUNCNAME ("BestXAllResourceUnlock [allresunlock]");
  BX_TRY_VARS_NO_PROG;
  bx_int32 current_port;

  bx_int32 ReslockReg;
  bx_int32 Mask=BX_EXERCISER_LOCK_MASK | BX_OBSERVER_LOCK_MASK |
                BX_ANALYZER_LOCK_MASK | BX_PERFORMANCE_LOCK_MASK;

  Mask= ( Mask<<BX_PORT_IS_RS232 ) | ( Mask<<BX_PORT_IS_PCI ) |
        ( Mask<<BX_PORT_IS_FHIF  ) | ( Mask<<BX_PORT_IS_USB );

  /* read kind of port to prevent error in case of OFFLINE */
  BX_TRY(BestXResourceLockDeterminePort(handle,&current_port));
  if( (current_port==BX_PORT_IS_OFFLINE) || (current_port==BX_PORT_IS_PRODUCTION))
       BX_ERRETURN(BX_E_OK);


  BX_TRY_BEGIN
  {
    BX_TRY(BestXDirectRegRead(handle,
                            BX_REG_RES_LOCK_REG,
                            sizeof(bx_int32),
                            &ReslockReg
                            ));

    ReslockReg&=(~Mask);
    BX_TRY(BestXDirectRegWrite (handle,
                              BX_REG_RES_LOCK_REG,
                              sizeof(bx_int32),
                              ReslockReg
                              ));
    BX_EXERCISER_RESLOCK_CTR=0;
    BX_OBSERVER_RESLOCK_CTR=0;
    BX_PERFORMANCE_RESLOCK_CTR=0;
    BX_ANALYZER_RESLOCK_CTR=0;
  } /*BX_TRY_BEGIN */

  BX_ERRETURN(BX_TRY_RET);

} /* BestXAllResourceUnlock */


/* -------------------------------------------------------------------------
* check if a resource is locked
* if the resource counter is zero even though the bit in the register is
* set, the value of the counter is inreased to 1
* ------------------------------------------------------------------------- */

bx_errtype EXPORT BestXResourceIsLocked(
  bx_handletype handle,
  bx_resourcetype resource,
  bx_int32 * lock_count,
  bx_porttype * lock_port)
{
  BX_DECLARE_FUNCNAME("BestXResourceIsLocked [resislocked]");
  BX_TRY_VARS_NO_PROG;
  bx_int8 outbuf[OUT_ISLOCKED];
  bx_int8ptr outptr = outbuf;
  bx_int16 outsize = OUT_ISLOCKED;
  bx_int32 ReslockReg;
  bx_int32 current_port;
  bx_int32 mask;
  bx_int32 bitfound;

  BX_HANDLECHECK;
  BX_FCT_PARAM_NULL_POINTER_CHK (lock_count);
  BX_FCT_PARAM_NULL_POINTER_CHK (lock_port);

  BX_TRY_BEGIN 
  {
    *lock_count = 0;
    *lock_port = BX_PORT_CURRENT;

    BX_TRY(BestXResourceLockDeterminePort(handle,&current_port));
    if( (current_port==BX_PORT_IS_OFFLINE) || (current_port==BX_PORT_IS_PRODUCTION))
       BX_ERRETURN(BX_E_OK);

    BX_TRY(BestXDirectRegRead(handle,
                            BX_REG_RES_LOCK_REG,
                            sizeof(bx_int32),
                            &ReslockReg
                            ));

    /* checks for which resource is asked then: 
       a) sets lock_count to the valid value 
       b) sets the mask for resource 
    */
    switch(resource)
    {
      case BX_RESLOCK_EXERCISER:
        *lock_count=BX_EXERCISER_RESLOCK_CTR;
        mask=BX_EXERCISER_LOCK_MASK;
        break;
      case BX_RESLOCK_OBSERVER:
        *lock_count=BX_OBSERVER_RESLOCK_CTR;
        mask=BX_OBSERVER_LOCK_MASK;
        break;
      case BX_RESLOCK_PERFORMANCE:
        *lock_count=BX_PERFORMANCE_RESLOCK_CTR;
        mask=BX_PERFORMANCE_LOCK_MASK;
        break;
      case BX_RESLOCK_ANALYZER:
        *lock_count=BX_ANALYZER_RESLOCK_CTR;
        mask=BX_ANALYZER_LOCK_MASK;
        break;
      default:
        BX_TRY_ERROR(BX_E_INVALID_CASE);
        break;
    } /*switch(resource)*/

     /* the mask for the resource is 'mask' now. We shift it to the 
     different positions in the Reslock Register to check if any 
     of the ports locked the resource 
     IMPORTANT: no distinction is made, whether BX_PORT_PCI_IO or 
     BX_PORT_PCI_CONF locked the resource  ! 
     BX_PORT_CURRENT is returned, when the current port equals the 
     port that locked the resource */

    bitfound=0; /* in case port is current port. *lock_port is not 
                   sufficient to indicate that, as it is set to 
                   BX_PORT_CURRENT per default !!! */
    
    if((mask << BX_PORT_IS_RS232) & ReslockReg)
    {
      *lock_port=BX_PORT_RS232;
      if(current_port==BX_PORT_IS_RS232)
      {
        *lock_port=BX_PORT_CURRENT;
        bitfound=1;
      }
    }
    
    if((mask << BX_PORT_IS_PCI) & ReslockReg)
    {
      *lock_port=BX_PORT_PCI_CONF;
      if(current_port==BX_PORT_IS_PCI) 
      {
        *lock_port=BX_PORT_CURRENT;
        bitfound=1;
      }
    }
    
    if((mask << BX_PORT_IS_FHIF) & ReslockReg)
    {
      *lock_port=BX_PORT_PARALLEL;
      if(current_port==BX_PORT_IS_FHIF) 
      {
        *lock_port=BX_PORT_CURRENT;
        bitfound=1;
      }
    }
    
    if((mask << BX_PORT_IS_USB) & ReslockReg)
    {
      *lock_port=BX_PORT_USB;
      if(current_port==BX_PORT_IS_USB)
      {
        *lock_port=BX_PORT_CURRENT;
        bitfound=1;
      }
    }

    /* in case the user canceled the program without reseting the card,
       the counter is in the db is 0 while the value in the register is 
       still valid. Solution: the counter is set to 1*/
    if( (*lock_count==0) && (bitfound) )
    {
      *lock_count=1;
      switch(resource)
      {
          case BX_RESLOCK_EXERCISER:
            BX_EXERCISER_RESLOCK_CTR=1;
            break;
          case BX_RESLOCK_OBSERVER:
            BX_OBSERVER_RESLOCK_CTR=1;
            break;
          case BX_RESLOCK_PERFORMANCE:
            BX_PERFORMANCE_RESLOCK_CTR=1;
            break;
          case BX_RESLOCK_ANALYZER:
            BX_ANALYZER_RESLOCK_CTR=1;
            break;
          default:
            BX_TRY_ERROR(BX_E_INVALID_CASE);
            break;
      } /*switch(resource)*/
    }

  } /* BX_TRY_BEGIN */
  BX_ERRETURN(BX_TRY_RET);
} /* BestXResourceIsLocked */


static bx_errtype BestXResourceLockInit(bx_handletype handle)
{
  /* this is a workaround to set the counters to one if there is a value */
  /* in the register */
  bx_int32 lock_count;
  bx_porttype lock_port;
  
  BestXResourceIsLocked(handle,BX_RESLOCK_EXERCISER,&lock_count,&lock_port);
  BestXResourceIsLocked(handle,BX_RESLOCK_OBSERVER,&lock_count,&lock_port);
  BestXResourceIsLocked(handle,BX_RESLOCK_PERFORMANCE,&lock_count,&lock_port);
  BestXResourceIsLocked(handle,BX_RESLOCK_ANALYZER,&lock_count,&lock_port);
  
  return BX_E_OK;
} /* BestXResourceLockInit(handle) */

/**************************************************************/

bx_errtype BestXAbstractPropByteSet(bx_handletype handle,
                                    bx_int16 cmdcode,
                                    bx_int8 prpid,
                                    bx_int32 prpval)
{
  BX_DECLARE_FUNCNAME("BestXAbstractPropByteSet");

  bx_errtype err;
  bx_int8 buf[2];
  bx_int8ptr bufptr = buf;
  bx_int8 value = (bx_int8) prpval;
  BX_FCT_PARAM_CHK(3, prpval > 0xff);

  bufptr = BestXByteCopy(bufptr, &prpid, 1UL);
  bufptr = BestXByteCopy(bufptr, &value, 1UL);
  err = BestXBasicCommand(handle, cmdcode, buf, 2, NULL, NULL);

  return err;
}


bx_errtype BestXAbstractPropShortSet(bx_handletype handle,
                                     bx_int16 cmdcode,
                                     bx_int8 prpid,
                                     bx_int32 prpval)
{
  BX_DECLARE_FUNCNAME("BestXAbstractPropShortSet");

  bx_errtype err;
  bx_int8 buf[3];
  bx_int8ptr bufptr = buf;
  bx_int16 value = (bx_int16) prpval;
  BX_FCT_PARAM_CHK(3, prpval > 0xffff);

  bufptr = BestXByteCopy(bufptr, &prpid, 1UL);
  bufptr = BestXWord2Stream(bufptr, &value, 1UL);
  err = BestXBasicCommand(handle, cmdcode, buf, 3, NULL, NULL);

  return err;
}


bx_errtype BestXAbstractPropLongSet(bx_handletype handle,
  bx_int16 cmdcode,
  bx_int8 prpid,
  bx_int32 prpval)
{
  bx_errtype err;
  bx_int8 buf[5];
  bx_int8ptr bufptr = buf;
  bufptr = BestXByteCopy(bufptr, &prpid, 1UL);
  bufptr = BestXLong2Stream(bufptr, &prpval, 1UL);
  err = BestXBasicCommand(handle, cmdcode, buf, 5, NULL, NULL);

  return err;
}


bx_errtype BestXAbstractPropByteGet(bx_handletype handle,
  bx_int16 cmdcode,
  bx_int8 prpid,
  bx_int32 * prpval)
{
  bx_errtype err;
  bx_int8 outbuf[1];
  bx_int16 outsize = 1;
  err = BestXBasicCommand(handle, cmdcode, &prpid, 1, outbuf, &outsize);
  *prpval = outbuf[0];
  return err;
}


bx_errtype BestXAbstractPropShortGet(
 bx_handletype handle,
 bx_int16 cmdcode,
 bx_int8 prpid,
 bx_int32 * prpval)
{
  bx_errtype err;
  bx_int8 outbuf[2];
  bx_int16 outsize = 2;
  bx_int16 value;
  err = BestXBasicCommand(handle, cmdcode, &prpid, 1, outbuf, &outsize);
  (void) BestXStream2Word(&value, outbuf, 1UL);
  *prpval = value;
  return err;
}


bx_errtype BestXAbstractPropLongGet(bx_handletype handle,
  bx_int16 cmdcode,
  bx_int8 prpid,
  bx_int32 * prpval)
{
  bx_errtype err;
  bx_int8 outbuf[4];
  bx_int16 outsize = 4;
  err = BestXBasicCommand(handle, cmdcode, &prpid, 1, outbuf, &outsize);
  (void) BestXStream2Long(prpval, outbuf, 1UL);
  return err;
}


/* --------------------------------------------------------------------------
* SCR; this series of helper functions simplify calls to BestXBasicCommand
* which send OR receive (NOT both) simple variables.
* NOTE; the IN_/OUT_ macros are not used....the stream lengths are implicit.
* -------------------------------------------------------------------------- */

/* SENDING simple vars...OUT_ macro must be 0 !!! */
bx_errtype BestXBBCSendByte(bx_handletype handle, bx_int16 cmdcode, bx_int8ptr pByte)
{
  bx_int8 cmdbuf[sizeof(bx_int8) + 1];
  (void) BestXByteCopy(cmdbuf, pByte, 1UL);
  return BestXBasicCommand(handle, cmdcode, cmdbuf, sizeof(bx_int8), NULL, NULL);
}

bx_errtype BestXBBCSendWord(bx_handletype handle, bx_int16 cmdcode, bx_int16 * pUWord)
{
  bx_int8 cmdbuf[sizeof(bx_int16) + 1];
  (void) BestXWord2Stream(cmdbuf, pUWord, 1UL);
  return BestXBasicCommand(handle, cmdcode, cmdbuf, sizeof(bx_int16), NULL, NULL);
}

bx_errtype BestXBBCSendLong(bx_handletype handle, bx_int16 cmdcode, bx_int32 * pULong)
{
  bx_int8 cmdbuf[sizeof(bx_int32) + 1];
  (void) BestXLong2Stream(cmdbuf, pULong, 1UL);
  return BestXBasicCommand(handle, cmdcode, cmdbuf, sizeof(bx_int32), NULL, NULL);
}
/* RECEIVING simple vars */
bx_errtype BestXBBCReceiveByte(bx_handletype handle, bx_int16 cmdcode, bx_int8ptr pByte)
{
  bx_int8 cmdbuf[sizeof(bx_int8) + 1] = {0};
  bx_int16 outsize = sizeof(bx_int8);
  bx_errtype err = BestXBasicCommand(handle, cmdcode, NULL, 0, cmdbuf, &outsize);
  (void) BestXByteCopy(pByte, cmdbuf, 1UL);
  return err;
}

bx_errtype BestXBBCReceiveWord(bx_handletype handle, bx_int16 cmdcode, bx_int16 * pUWord)
{
  bx_int8 cmdbuf[sizeof(bx_int16) + 1] = {0};
  bx_int16 outsize = sizeof(bx_int16);
  bx_errtype err = BestXBasicCommand(handle, cmdcode, NULL, 0, cmdbuf, &outsize);
  (void) BestXStream2Word(pUWord, cmdbuf, 1UL);
  return err;
}

bx_errtype BestXBBCReceiveLong(bx_handletype handle, bx_int16 cmdcode, bx_int32 * pULong)
{
  bx_int8 cmdbuf[sizeof(bx_int32) + 1] = {0};
  bx_int16 outsize = sizeof(bx_int32);
  bx_errtype err = BestXBasicCommand(handle, cmdcode, NULL, 0, cmdbuf, &outsize);
  (void) BestXStream2Long(pULong, cmdbuf, 1UL);
  return err;
}

/*---------------------------------------------------------
* xcapi_version returns the version number
*---------------------------------------------------------*/
bx_vertype  EXPORT * xcapi_version(bx_int32 type)
{
  /* type=0 : return CAPI   version from CAPI
     type=1 : return BIOS   version from CAPI
     type=2 : return BIOS2X version from CAPI
  */

  static bx_vertype CapiVersion=BX_VER_INIT(BX_VER_CAPI);
  static bx_vertype BiosVersion=BX_VER_INIT(BX_VER_BIOS);
  static bx_vertype Bios2XVersion=BX_VER_INIT(BX_VER_BIOS2X);

  switch (type)
  {
    case 0:
      return (&CapiVersion);
      break;  
    case 1:
      return (&BiosVersion);
      break;  
    case 2:
      return (&Bios2XVersion);
      break;  
    default:
      return (&CapiVersion);
      break;
  }
}

#ifndef BEST_FIRMWARE

/* --------------------------------------------------------------------------
* Use these functions for ALL byte-ordering.
* They are tested safe for (dest == src) on MSVC (Win32) and Borland 4.52
* NOTE; BESTX_MEMCPY IS NOT always safe for dest==src (compiler dependent) !!!
* -------------------------------------------------------------------------- */

/* --------------------------------------------------------------------------
* NOTE; Non-linear memory models will be limited i.e. DOS; size_t == UInt16
* -------------------------------------------------------------------------- */


/******************************************************************************/
bx_int8ptr EXPORT BestXByteCopy(bx_int8ptr dest, bx_int8ptr src, bx_int32 count)
{
  if (dest != src)
  {
    BESTX_MEMCPY(dest, src, (size_t) count);
  }
  return (dest + (size_t) count);
}


/******************************************************************************/
typedef struct _WORD2STREAM
{
  bx_int8 AsBytes[2];
} WORD2STREAM;


/******************************************************************************/
bx_int8ptr EXPORT BestXWord2Stream(bx_int8ptr dest, bx_int16ptr src, bx_int32 count) /* num_words */
{
  WORD2STREAM tmp;
  if (BestXIsHostLittleEndian())
  {
    while (count--)
    {
      tmp = *(WORD2STREAM *) src;
      *dest++ = tmp.AsBytes[1];
      *dest++ = tmp.AsBytes[0];
      src++;
    }
  }
  else
  {
    if (dest != (bx_int8ptr) src)
    {
      BESTX_MEMCPY(dest, src, (size_t) (count * sizeof(bx_int16)));
      dest += (size_t) (count * sizeof(bx_int16));
    }
  }

  return dest;
}


/******************************************************************************/
bx_int8ptr EXPORT BestXStream2Word(bx_int16ptr dest, bx_int8ptr src, bx_int32 count) /* num_words */
{
  if (BestXIsHostLittleEndian())
  {
    while (count--)
    {
      *dest = (bx_int16) (((bx_int16) src[0] << 8) | (bx_int16) src[1]);
      dest++;
      src += 2;
    }
  }
  else
  {
    if ((bx_int8ptr) dest != src)
    {
      BESTX_MEMCPY(dest, src, (size_t) (count * sizeof(bx_int16)));
      src += (size_t) (count * sizeof(bx_int16));
    }
  }

  return src;
}


/******************************************************************************/
typedef struct _LONG2STREAM
{
  bx_int8 AsBytes[4];
} LONG2STREAM;


/******************************************************************************/
bx_int8ptr EXPORT BestXLong2Stream(bx_int8ptr dest, bx_int32ptr src, bx_int32 count) /* num_dwords */
{
  LONG2STREAM tmp;
  if (BestXIsHostLittleEndian())
  {
    while (count--)
    {
      tmp = *(LONG2STREAM *) src;
      *dest++ = tmp.AsBytes[3];
      *dest++ = tmp.AsBytes[2];
      *dest++ = tmp.AsBytes[1];
      *dest++ = tmp.AsBytes[0];
      src++;
    }
  }
  else
  {
    if (dest != (bx_int8ptr) src)
    {
      BESTX_MEMCPY(dest, src, (size_t) (count * sizeof(bx_int32)));
      dest += (size_t) (count * sizeof(bx_int32));
    }
  }

  return dest;
}


/******************************************************************************/
bx_int8ptr EXPORT BestXStream2Long(bx_int32ptr dest, bx_int8ptr src, bx_int32 count) /* num_dwords */
{
  if (BestXIsHostLittleEndian())
  {
    while (count--)
    {
      *dest = ((bx_int32) src[0] << 24) |
        ((bx_int32) src[1] << 16) |
        ((bx_int32) src[2] << 8) |
        (bx_int32) (src[3]);
      dest++;
      src += 4;
    }
  }
  else
  {
    if ((bx_int8ptr) dest != src)
    {
      BESTX_MEMCPY(dest, src, (size_t) (count * sizeof(bx_int32)));
      src += (size_t) (count * sizeof(bx_int32));
    }
  }

  return src;
}

#endif /* not firmware */


/* TODO: Move this function to internal.c but remove the call from
aprpdefload to have a linkable source code delivery. CZ */

/* ------------------------------------------------------------------------- *
* This functions sets the debug properties on board of the e2926a. If the   *
* hardware is not e2926, this call is ignored.                              *
* ------------------------------------------------------------------------- */

bx_errtype EXPORT BestXDebugPropSet(bx_handletype handle,
                                    bx_debugtype prop,
                                    bx_int32 value)
{
  BX_DECLARE_FUNCNAME ("BestXDebugPropSet [dbgprpset]");
  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
  /* depending on the value of 'value' the debug property is either set
    * or not. Different commands are called. (see below). */
    if(BestXHasFirmware(handle))
    {
      if (prop == BX_FW_ALL)
      {
        /* set or clear all is the command */
        BX_TRY(BestXBasicCommand(handle,
          (bx_int16) (value ? CMD_DBG_SECTION_SETALL :
        CMD_DBG_SECTION_CLEARALL),
          NULL, 0, NULL, NULL));
      }

      else
      {
        bx_int8 inbuf = (bx_int8) prop;
        BX_TRY(BestXBasicCommand(handle,
          (bx_int16) (value ? CMD_DBG_SECTION_SET :
        CMD_DBG_SECTION_CLEAR),
          &inbuf, 1, NULL, NULL));
      }
    }
  }


  BX_ERRETURN(BX_TRY_RET);
}

/******************************************************************************/
bx_errtype EXPORT BestXDirectRegWrite(
  bx_handletype handle,
  bx_int32 dir_addr,
  bx_int32 regsize,
  bx_int32 reg_value
)
{
  bx_errtype err=BX_E_OK;
  bx_int8 zw[16];
  bx_int8ptr bp;
  bx_int8 size;

#ifdef BEST_DEBUG
  /* automatically open debug file */
  if (DirectFP[handle]==NULL)
  {
    char fname[256];
    sprintf(fname,"%s%lu.txt",DIRECT_REPORT_FILE,handle);
    BestXDirectRegLogOpen (handle, fname);
  }
#else
  /* Release version:
     Open file only on demand: Call BestXDirectRegLogOpen()
  */
#endif

  if (DirectFP[handle])
  {
    BESTX_FPRINTF(DirectFP[handle],"drw dad=%06lx\\h val=%08lx\\h size=%lu\n",
      dir_addr,reg_value,regsize);
    fflush(DirectFP[handle]);
  }

  switch (bx_handlearray[handle].port)
  {
   case BX_PORT_OFFLINE:
   {
     err=BX_E_OK;
     break;
   }
   case BX_PORT_FASTHIF:
   case BX_PORT_USB:
   case BX_PORT_PARALLEL:
   case BX_PORT_RS232:

     if (BestX16BitRegisterFile(handle))
   {
     bp = BestXLong2Stream(zw, &dir_addr, 1); /* store the address */
     size=(bx_int8) regsize;
     bp = BestXByteCopy(bp, &size, 1);  /* and the size  */

     (void) BestXLong2Stream(bp, &reg_value, 1);  /* and the value  */

     err = BestXBasicCommand(handle, CMD_REGISTER_WRITE, zw,
           IN_REGISTER_WRITE, NULL, NULL);
     }
     else
     {
       bp = BestXByteCopy(zw, (bx_int8ptr)&dir_addr,4); /* store the address */
            BestXByteCopy(bp, (bx_int8ptr)&regsize ,4); /* store the size    */
       err=BestXBasicBlockWrite(handle,DIRECT_ADDRESS,zw,8);
 
       if (err==BX_E_OK)
       {
         err=BestXBasicBlockWrite(handle,DIRECT_DATA,(bx_int8ptr)&reg_value,4);
       }
     }
     break;

#ifdef CUSTOM_OEM1
   case BX_PORT_OEM:
#endif
   case BX_PORT_PCI_CONF:
   case BX_PORT_PCI_IO:
   {
     err=BestXDirectRegPCIWriteEx(handle,dir_addr,regsize,reg_value);
     break;
   }

   case BX_PORT_PRODUCTION:
     if (bx_handlearray[handle].hwinfo.hw==BX_HW_E2923A ||
         bx_handlearray[handle].hwinfo.hw==BX_HW_E2923B ||
         bx_handlearray[handle].hwinfo.hw==BX_HW_E2923F)
     {
       /* E2923 */
#ifndef LINUX
       err=(bx_errtype)!BestXDRW23(bx_handlearray[handle].portnumber, dir_addr,regsize,reg_value);
 #endif
     }
     else
     {
       /* E2922 */
       err=BestXDirectRegHostWrite((bx_handletype) bx_handlearray[handle].portnumber,
                                  (bx_handletype)bx_handlearray[handle].entered_port, /*Device on Backplane */
                                  dir_addr,
                                  regsize,
                                  reg_value);
     }
     break;


   default:
    err=BX_E_INVALID_CASE;
  }  /*lint !e788 */
    
  BX_ERRETURN (err);
}

/* This function should be called instead of BestXDirectRegPCIWrite().
   It contains a Mephisto bugfix/workaround for the observer mask registers
*/

static bx_errtype BestXDirectRegPCIWriteEx(
  bx_handletype handle,
  bx_int32 dir_addr,
  bx_int32 regsize,    
  bx_int32 reg_value
)
{
  BX_DECLARE_FUNCNAME("BestXDirectRegPCIWriteEx");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    if (
        BestXHasMephisto(handle) &&   
        (dir_addr==BX_REG_PROT_ERR_MASK_M ||
         dir_addr==BX_REG_PROT_ERR_MASK_M+4) 
       )
    {
      bx_int32 val=0;
      bx_int32 MaxTries=10;
      bx_int32 reg_mask;
  
      /* CAUTION: We have to write twice, 
                  because of a bug in Mephisto.
                  In between, there is no other write access to 
                  Mephisto allowed (only problem when FW present)
                  so we need polling here ! */
      if (dir_addr == BX_REG_PROT_ERR_MASK_M+4) /* mask out illegal bits */
      {
        reg_mask = 0x1fffff;
      }
      else
      {
        reg_mask = (bx_int32) -1;
      }

      do 
      {
        BX_TRY(BestXDirectRegPCIWrite(handle,dir_addr,regsize,reg_value));
        BX_TRY(BestXDirectRegPCIWrite(handle,dir_addr,regsize,reg_value));
        BX_TRY(BestXDirectRegPCIRead(handle,dir_addr,regsize,&val));
      } while ( ((val & reg_mask) != (reg_value & reg_mask)) && --MaxTries);
    
      if (!MaxTries)
      {
        /* We were not able to write the register */
        BX_E_ERROR_MSG_SET("BestXDirectRegPCIWriteEx: Could not write observer mask !")
        BX_TRY_FAIL(BX_E_ERROR);
      }
    }
    else
    {
      /* we usually go here */
      BX_TRY(BestXDirectRegPCIWrite(handle,dir_addr,regsize,reg_value));
    }
  }

  BX_ERRETURN (BX_TRY_RET);
}


/*---------------------------------------------------------------------------*
 * BestXDirectRegBlockWrite
 *
 * Purpose: Transfers a block of bytes, words or DWs from host to card
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXDirectRegBlockWrite(
  bx_handletype handle,
  bx_int32      address,  /* Card address               */
  bx_int32      regsize,  /* 4 or 2 bytes per access    */
  bx_int32      autoinc,  /* Wether HW or SW increments */
  bx_int8ptr    data,     /* data to transfer           */
  bx_int32      datatype, /* format of data[]: 1,2,4    */
  bx_int32      numbytes  /* size of data in bytes      */
)
{
  BX_DECLARE_FUNCNAME("BestXDirectRegBlockWrite []");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN  
  {
    bx_int8ptr pStreamdata;
    /* Parameter regsize has to be 2 or 4 (DBI bus access width).
       Parameter datatype has to be 1,2 or 4, depending on wether
       data[] consists of bytes, words or DWs
       (handling depends on the host's endianess !).

       Currently implemented allowed combinations of
       {regsize,datatype} are:
       {2,1} (e.g. expansion ROM reading)
       {2,2} (e.g. behavior memories),
       {4,1} (e.g. datamemory),
       {4,4} (e.g. block memory, config-space)

       To be done (as needed): {2,4} and {4,2}
    */

    BX_TRY_FCT_PARAM_ALIGNMENT(numbytes, regsize);
    BX_TRY_FCT_PARAM_ALIGNMENT(numbytes, datatype);

    switch (bx_handlearray[handle].port)
    {
      case BX_PORT_OFFLINE:
      {
        break;
      }
      case BX_PORT_USB:
      case BX_PORT_FASTHIF:
      case BX_PORT_PARALLEL:
      case BX_PORT_RS232:
      {
        DataBufferResize(handle, numbytes);
        pStreamdata = bx_handlearray[handle].m_pBuffer;

        switch (datatype)
        {
          case sizeof(bx_int8): /* byte -> nothing to do here, just copy */
            memcpy (pStreamdata, data, numbytes);
            break;
          case sizeof(bx_int16): /* word ->  */
            BestXWord2Stream(pStreamdata, (bx_int16ptr) data,
              numbytes/sizeof(bx_int16));
            break;
          case sizeof(bx_int32): /* dword -> convert */
            BestXLong2Stream(pStreamdata, (bx_int32ptr) data,
              numbytes/sizeof(bx_int32));
            break;
          default:
            BX_ERRETURN(BX_E_INVALID_CASE); /* should not happen */
        }


       BX_TRY(BestXBasicBlockCommand( handle,
          address,
          numbytes,
          (bx_int8)(regsize & 0xFFUL),
          CMD_WRITE,
          autoinc ? BX_TRUE : BX_FALSE,
          pStreamdata ));
        break;
      }

#ifdef CUSTOM_OEM1
    case BX_PORT_OEM:
#endif
    case BX_PORT_PRODUCTION:
    case BX_PORT_PCI_CONF:
    case BX_PORT_PCI_IO:
    {
      bx_int32 t,NumTransfers;
      bx_int32 val32;
      bx_int16 val16;

      NumTransfers=numbytes/regsize;

      for (t=0;t<NumTransfers;t++)
      {
        /* Caution: val has to have endianess of host for BestXDirectRegWrite() */
        if (regsize==sizeof(bx_int32))
        {
          /* compute val32 for DirectRegisterWrite() */
          switch (datatype)
          {
          case sizeof(bx_int8): /* 1 */
            /* data[] is a byte array (always big-endianess).
            Need to convert to the host's-endianess
            (on Intel platforms a conversion from big-
            to little endianess takes place here):
            */
            BestXStream2Long(&val32,data+t*regsize,1);
            break;

          case sizeof(bx_int32): /* 4 */
            /* data[] is a DW array (host endianess).
            No conversion needed here because val32
            is also in host-endianess
            */
            val32= *((bx_int32ptr)(data+t*regsize));
            break;

          case sizeof(bx_int16): /* 2 */
          default:
            BX_ERRETURN(BX_E_INVALID_CASE); /* should never happen */
            break;
          }
        }
        else
        {
          assert(regsize==sizeof(bx_int16));

          /* compute val16 resp val32 for DirectRegisterWrite() */
          switch (datatype)
          {
          case sizeof(bx_int8): /* 1 */
            /* data[] is a byte array (always big-endianess).
            Need to convert to the host's-endianess
            (on Intel platforms a conversion from big-
            to little endianess takes place here):
            */
            BestXStream2Word(&val16,data+t*regsize,1);
            break;
          case sizeof(bx_int16): /* 2 */
            /* data[] is a word array (host endianess).
            No conversion needed here because val
            is also in host-endianess
            */
            val16= *((bx_int16ptr)(data+t*regsize));
            break;
          case sizeof(bx_int32): /* 4 */
          default:
            BX_ERRETURN(BX_E_INVALID_CASE); /* should never happen */
            break;
          }

          val32=val16;
        }

        BX_TRY(BestXDirectRegWrite(handle,
          address + (autoinc?0:t*regsize),
          regsize,
          val32
          ));
      }
      break;
    } /* case BX_PORT_PCI_IO */

    default:
      BX_ERRETURN(BX_E_INVALID_CASE);
    }  /* switch port */
  }
  BX_ERRETURN(BX_TRY_RET);
}


/*---------------------------------------------------------------------------*
 * bx_errtype EXPORT BestXDirectRegLogOpen (const char * filename)
 *
 * Purpose      : open log file for register accesses,
 *                closes file if filename == null
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXDirectRegLogOpen (bx_handletype handle,
                                              const char * filename)
{
  bx_errtype err = BX_E_OK;

  if (DirectFP[handle] != NULL)
  {
    fclose(DirectFP[handle]);
    DirectFP[handle] = NULL;
  }

  if ( filename != NULL && (BESTX_STRCMP(filename, "NULL") != 0) )
  {
    DirectFP[handle] = BESTX_FOPEN(filename, "wb"); /* open in binary mode, no CRs! */
    if (DirectFP[handle] == NULL)
    {
      BestXLastErrorParamStringSet(handle, filename);
      err = BX_E_FILE_OPEN;
    }
  }

  return BX_E_OK;
}

/*****************************************************************************/
bx_errtype EXPORT BestXDirectRegRead(
  bx_handletype handle,
  bx_int32 dir_addr,
  bx_int32 regsize,
  bx_int32 * reg_value
)
{
  bx_int8 in_zw[8];
  bx_int8 out_zw[OUT_REGISTER_READ];
  bx_errtype err;
  bx_int16 len;
  bx_int8ptr bp;
  bx_int8 size;


#ifdef BEST_DEBUG
  /* automatically open debug file */
  if (DirectFP[handle]==NULL)
  {
    char fname[256];
    sprintf(fname,"%s%lu.txt",DIRECT_REPORT_FILE,handle);
    BestXDirectRegLogOpen (handle, fname);
  }
#else
  /* Release version:
     Open file only on demand: Call BestXDirectRegLogOpen()
  */
#endif

  if (DirectFP[handle])
  {
    BESTX_FPRINTF(DirectFP[handle],"drr dad=%06lx\\h size=%lu\n",
      dir_addr,regsize);
    fflush(DirectFP[handle]);
  }

   switch (bx_handlearray[handle].port)
   {
      case BX_PORT_OFFLINE:
      {
          *reg_value=0;
          err=BX_E_OK;
      break;
      }
      case BX_PORT_FASTHIF:
      case BX_PORT_USB:
      case BX_PORT_PARALLEL:
      case BX_PORT_RS232:
        if (BestX16BitRegisterFile(handle))
      {
        bp = BestXLong2Stream(in_zw, &dir_addr, 1);  /* into the bytestream */
        size=(bx_int8) regsize;
        (void) BestXByteCopy(bp, &size, 1);   /* into the bytestream */
          len = OUT_REGISTER_READ;

          err = BestXBasicCommand(handle, CMD_REGISTER_READ, in_zw, IN_REGISTER_READ, out_zw, &len);

        (void) BestXStream2Long(reg_value, out_zw, 1);
        }
        else
        {
          /* This only works on little endian machines !!! 
             We would need a BestXStream2Long() working the other way round ! */
          bp=BestXByteCopy(in_zw,(bx_int8ptr)&dir_addr,4);
          BestXByteCopy(bp,(bx_int8ptr)&regsize,4);
          err=BestXBasicBlockWrite(handle,DIRECT_ADDRESS,in_zw,8);
          
          if (err==BX_E_OK)
          {
            err=BestXBasicBlockRead(handle,DIRECT_DATA,(bx_int8ptr)reg_value,4);
          }
        }
      break;

#ifdef CUSTOM_OEM1
      case BX_PORT_OEM:
#endif
      case BX_PORT_PCI_CONF:
      case BX_PORT_PCI_IO:
      {
        err=BestXDirectRegPCIRead(handle,dir_addr,regsize,reg_value);
        break;
      }

     case BX_PORT_PRODUCTION:
       {
         if (bx_handlearray[handle].hwinfo.hw==BX_HW_E2923A ||
             bx_handlearray[handle].hwinfo.hw==BX_HW_E2923B ||
             bx_handlearray[handle].hwinfo.hw==BX_HW_E2923F)
         {
           /* E2923 */
#ifndef LINUX
           err=(bx_errtype)!BestXDRR23(bx_handlearray[handle].portnumber, dir_addr,regsize,reg_value);
#endif
         }
         else
         {
           err=BestXDirectRegHostRead((bx_handletype)bx_handlearray[handle].portnumber,
                                      (bx_handletype)bx_handlearray[handle].entered_port, /*Device on Backplane */
                                      dir_addr,
                                      regsize,
                                      reg_value);
         } 
         break;
       }
      
      default:
        err=BX_E_INVALID_CASE;
  } /* switch */

  if (DirectFP[handle])
  {
    BESTX_FPRINTF(DirectFP[handle],"# val=%lx\\h\n",*reg_value);
    fflush(DirectFP[handle]);
  }

  BX_ERRETURN (err);
}

/*---------------------------------------------------------------------------*
 * BestXDirectRegFaustFlashRead()
 *
 * Purpose: Reads two words from Faust flash memory
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXDirectRegFaustFlashRead(   /* @drffr */
  bx_handletype handle,
  bx_int32 addr,     /* @dad, a "DW-flash address" */
  bx_int32 *value    /* #RETURN "0x%08lx" */
)
{
  bx_int32 valchip0,valchip1;
  BX_TRY_VARS_NO_PROG; 

  /* Parameter addr must have the following format (compare xaddrmap.h):
     addr[31-24] = 0
     addr[23-20] = A[21-18] = PAGE[3-0]
     addr[19-2]  = A[17-0]  = DBI[19-2]
     addr[1-0]   = 0
    
     Examples:
     Image 0: addr=0x000000 => PAGEPTR=0x0 A=0x000000
              addr=0x000004 => PAGEPTR=0x0 A=0x000001
              addr=0x000008 => PAGEPTR=0x0 A=0x000002
              addr=0x0ffffc => PAGEPTR=0x0 A=0x03ffff
              addr=0x100000 => PAGEPTR=0x1 A=0x000000
              addr=0x1ffffc => PAGEPTR=0x1 A=0x03ffff
              addr=0x200000 => PAGEPTR=0x2 A=0x000000
              addr=0x2ffffc => PAGEPTR=0x2 A=0x03ffff
              addr=0x300000 => PAGEPTR=0x3 A=0x000000
              addr=0x3ffffc => PAGEPTR=0x3 A=0x03ffff
     Image 1: addr=0x400000 => PAGEPTR=0x4 A=0x100000
     Image 2: addr=0x800000 => PAGEPTR=0x8 A=0x200000
     Image 3: addr=0xc00000 => PAGEPTR=0xc A=0x300000
              addr=0xfffffc => PAGEPTR=0xf A=0x03ffff
   */

  BX_TRY_BEGIN
  {
    /* We got 16MB of memory althogether; last address is 0xfffffc */
    if (addr>=0x1000000 || addr%4 || value==NULL)
    {
      BX_E_ERROR_MSG_SET("Address must be less than 0x1000000 and on a DW boundary");
      BX_TRY_FAIL(BX_E_ERROR);      
    }

    /* Set A[21-18] */
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_COCO_PAGEPTR,sizeof(bx_int16),(addr>>20) & 0xf));

    /* Read word from chip0 */
    BX_TRY(BestXDirectRegRead(handle,FLASH2DBI(((addr>>2)&0x3ffff),0),sizeof(bx_int16),&valchip0));

    /* Read word from chip1 */
    BX_TRY(BestXDirectRegRead(handle,FLASH2DBI(((addr>>2)&0x3ffff),1),sizeof(bx_int16),&valchip1));

    /* e.g. addr=0, *value=0x11223344 means: 
       first byte of faust-image is 0x11 
       second byte of mephisto-image is 0x22 etc.
    */
    
    *value= ((valchip0<<16)&0xffff0000)|(valchip1&0xffff); 
  }
  BX_ERRETURN(BX_TRY_RET);
}


/*---------------------------------------------------------------------------*
 * bx_errtype EXPORT BestXDirectBlockRead(
 *
 * Purpose  : changed parameter order for CLI purposes
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXDirectBlockRead(
  bx_handletype handle,
  bx_int32      address,
  bx_int32      regsize,
  bx_int32      autoinc,
  bx_int32      datatype,
  bx_int32      numbytes,
  bx_int8 *     data
)
{
  return BestXDirectRegBlockRead(handle, address, regsize, autoinc, data, datatype, numbytes);
}

/*---------------------------------------------------------------------------*
 * BestXDirectRegBlockRead
 *
 * Purpose: Transfers a block of bytes, words or DWs from card to host
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXDirectRegBlockRead(
  bx_handletype handle,
  bx_int32      address,  /* Card address               */
  bx_int32      regsize,  /* 4 or 2 bytes per access    */
  bx_int32      autoinc,  /* Wether HW or SW increments */
  bx_int8ptr    data,     /* data to transfer           */
  bx_int32      datatype, /* format of data[]: 1,2,4    */
  bx_int32      numbytes  /* size of data in bytes      */
)
{
  BX_DECLARE_FUNCNAME("BestXDirectRegBlockRead []");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN 
  {
    /* Parameter regsize has to be 2 or 4 (DBI bus access width).
       Parameter datatype has to be 1,2 or 4, depending on wether
       data[] consists of bytes, words or DWs
       (handling depends on the host's endianess !).

       Currently implemented allowed combinations of
       {regsize,datatype} are:
       {2,1} (e.g. expansion ROM reading)
       {2,2} (e.g. behavior memories),
       {2,4} (e.g. Faust trace memory)
       {4,1} (e.g. datamemory),
       {4,4} (e.g. block memory, config-space, Mephisto trace memory)

       To be done (as needed): {4,2}
    */

    BX_TRY_FCT_PARAM_ALIGNMENT(numbytes, regsize);
    BX_TRY_FCT_PARAM_ALIGNMENT(numbytes, datatype);

    switch (bx_handlearray[handle].port)
    {
      case BX_PORT_OFFLINE:
      {
        memset (data, '\0', numbytes);
        break;
      }
      case BX_PORT_USB:
      case BX_PORT_FASTHIF:
      case BX_PORT_PARALLEL:
      case BX_PORT_RS232:
      {
        BX_TRY(BestXBasicBlockCommand( handle,address,numbytes,(bx_int8) (regsize & 0xffUL),CMD_READ,autoinc ? BX_TRUE : BX_FALSE,data));
        switch (datatype)
        {
          case sizeof(bx_int8): /* byte -> nothing to do here */
            break;
          case sizeof(bx_int16): /* word ->  */
            BestXStream2Word((bx_int16ptr) data, data,numbytes / sizeof(bx_int16));
            break;
          case sizeof(bx_int32): /* dword -> convert */
            BestXStream2Long((bx_int32ptr) data, data,numbytes / sizeof(bx_int32));
            break;
          default:
            BX_ERRETURN(BX_E_INVALID_CASE); /* invalid par, should not happen */
        }
        break;
      }
     
#ifdef CUSTOM_OEM1
      case BX_PORT_OEM:
#endif
      case BX_PORT_PRODUCTION:
      case BX_PORT_PCI_CONF:
      case BX_PORT_PCI_IO:
      {
        bx_int32 t, NumTransfers;
        bx_int32 val32;
        bx_int16 val16;
       
        NumTransfers=numbytes/regsize;

        for (t=0;t<NumTransfers;t++)
        {
          BX_TRY(BestXDirectRegRead(handle,address + (autoinc?0:t*regsize),regsize,&val32 /* points to _4_ bytes !! */));
          /* Caution: val has endianess of host, data is big endian */
          if (regsize==sizeof(bx_int32))
          {
            switch (datatype)
            {
              case sizeof(bx_int8): /* 1 */
                /* data[] is a byte array (always big-endianess).
                    Need to convert from the host's-endianess to
                    big-endianess
                    (on Intel platforms a conversion from little-
                    to big endianess takes place here): 
                */
                BestXLong2Stream(data+t*regsize, &val32, 1UL);
                break;
              case sizeof(bx_int32): /* 4 */
                /* data[] is a DW array (host endianess).
                   No conversion needed here because val32
                   is also in host-endianess   
                */
                *((bx_int32ptr)(data+t*regsize))=val32;
                break;
              case sizeof(bx_int16): /* 2 */
              default:
                BX_ERRETURN(BX_E_INVALID_CASE); /* should never happen */
                break;
            }
          }
          else
          {
            assert(regsize==sizeof(bx_int16));
            val16=(bx_int16) val32;
            switch (datatype)
            {
              case sizeof(bx_int8): /* 1 */
                /* data[] is a byte array (always big-endianess).
                   Need to convert from the host's-endianess to
                   big-endianess
                   (on Intel platforms a conversion from little-
                   to big endianess takes place here):
                */
                BestXWord2Stream(data+t*regsize,&val16,1);
                break;
              case sizeof(bx_int16): /* 2 */
                /* data[] is a word array (host endianess).
                   No conversion needed here because val16
                   is also in host-endianess   */
                *((bx_int16ptr)(data+t*regsize))=val16;
                break;
              case sizeof(bx_int32): /* 4 */
                /* We go her e.g. when reading tracedata from
                   a Faust based HW via the PCI port.
                   data[] is a dword array (host endianess).
                   The bytes in the word val16 are also in host
                   endianess; need only to take care of
                   the word-odering in the DW array data[].
                   */
                   
                if (BestXIsHostLittleEndian())
                {
                  /* Need to swap the two words in the dword array */
                  if (t%2)
                  {
                    *((bx_int16ptr)(data+(t-1)*regsize))=val16;
                  }
                  else
                  {
                    *((bx_int16ptr)(data+(t+1)*regsize))=val16;
                  }
                }
                else
                {
                  *((bx_int16ptr)(data+(t+1)*regsize))=val16;
                }
                break;
              default:
                BX_ERRETURN(BX_E_INVALID_CASE); /* should never happen */
                break;
            } /* switch */
          } /* else */
        } /*for */
      } /* case PCI port */
      break;
      default:
        BX_ERRETURN(BX_E_INVALID_CASE);
    }  /* switch */
  } /* TRY */
  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * bx_errtype EXPORT BestXDirectRegMaskWrite
 *
 * Purpose      : set specific bits in register, not touching masked ones.
 *                only bits where mask is '1' are changed!
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXDirectRegMaskWrite ( bx_handletype handle,
                                           bx_int32 address,
                                           bx_int32 regsize,
                                           bx_int32 mask,
                                           bx_int32 value )
{
  BX_DECLARE_FUNCNAME ("BestXDirectRegMaskWrite [drmw]");
  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    bx_int32 oldvalue;
    BX_TRY(BestXDirectRegRead (handle, address, regsize, &oldvalue));

    value = (oldvalue & ~mask) | (value & mask);

    BX_TRY(BestXDirectRegWrite (handle, address, regsize, value));
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*****************************************************************************
 * static functions
 *****************************************************************************/
/*---------------------------------------------------------------------------*
 * static void DataBufferDelete(handle)
 *
 * Purpose      : get rid of general purpose buffer
 *-----------------------------------------------------------*/
static void DataBufferDelete(bx_handletype handle)
{
  if (bx_handlearray[handle].m_bufferSize > 0)
  {
    BestXMemFree((void **) &(bx_handlearray[handle].m_pBuffer));
    bx_handlearray[handle].m_bufferSize = 0UL;
  }
}

/*---------------------------------------------------------------------------*
 * static void DataBufferResize(handle, bx_int32 size)
 *
 * Purpose      : make buffer with new size, but only if new size is
 *                bigger than old one
 *---------------------------------------------------------------------------*/
static void DataBufferResize(bx_handletype handle, bx_int32 size)
{
  if (size > bx_handlearray[handle].m_bufferSize)
  {
    bx_handlearray[handle].m_pBuffer =
      BestXMemRealloc((void *) (bx_handlearray[handle].m_pBuffer),
                      (size_t) size);

    if (bx_handlearray[handle].m_pBuffer != NULL)
    {
      bx_handlearray[handle].m_bufferSize = size;
    }
    else
    {
      bx_handlearray[handle].m_bufferSize = 0UL;
    }
  }
}
